<?php

declare(strict_types=1);

namespace Tanzsport\ExcelTeamListReader;

use InvalidArgumentException;
use PHPUnit\Framework\TestCase;
use Tanzsport\ExcelTeamListReader\Model\TeamList;

class TeamListsExcelReaderTest extends TestCase
{
    private $expected;

    public function setUp(): void
    {
        parent::setUp();
        $this->expected = $this->loadExpected();
    }

    public function testReader()
    {
        $files = array('Bundeskaderlisten.xlsx', 'Bundeskaderlisten.xls');
        $namen = array('Kombination' => 3, 'Latein' => 28, 'Standard' => 21, 'Formationen' => 6, 'DRBV' => 4, 'Professional Division' => 2);

        foreach ($files as $file) {
            $reader = $this->createReader($file);
            $listen = $reader->getTeamLists();
            $this->assertNotNull($listen);
            $this->assertEquals(6, count($listen));

            foreach ($namen as $name => $anzahl) {
                $liste = $reader->findTeamList($name);
                $this->assertNotNull($liste, "Liste geladen: {$file}");
                $this->assertEquals($anzahl, count($liste->getEintraege()), "Anzahl stimmt: {$file} => {$name} = {$anzahl}");
                $this->assertExpected($name, $liste);
                $this->assertTrue(count($liste->getKopfzeile()) == 5);
            }
        }
    }

    public function testInvalidFile()
    {
        $this->expectException(InvalidArgumentException::class);
        $this->expectExceptionMessage("gültige Excel-Datei");

        $reader = $this->createReader('invalid.jpg');
        $reader->getTeamLists();
    }

    private function createReader(string $file)
    {
        return new ExcelTeamListReader(__DIR__ . '/' . $file);
    }

    private function loadExpected()
    {
        $json = file_get_contents(__DIR__ . '/expected.json');
        return json_decode($json);
    }

    private function assertExpected($name, TeamList $liste)
    {
        if (isset($this->expected->$name)) {
            foreach ($this->expected->$name as $ex) {
                $eintraege = $liste->getEintraege();
                if (isset($eintraege[$ex->index])) {
                    $eintrag = $eintraege[$ex->index];
                    $vars = (array)$ex->values;
                    foreach ($vars as $var => $expectedValue) {
                        $this->assertEquals($expectedValue, $eintrag->$var, "Wert stimmt nicht überein: {$name}, {$ex->index} => {$var} = {$expectedValue}");
                    }
                } else {
                    $this->fail("Eintrag nicht vorhaden: {$name}, index={$ex->index}");
                }
            }
        }
    }
}
