<?php

declare(strict_types=1);

namespace Tanzsport\ExcelTeamListReader;

use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use Tanzsport\ExcelTeamListReader\Model\TeamList;

class ExcelTeamListReader
{
    private string $path;

    /**
     * @var TeamList[]
     */
    private ?array $teamLists = null;

    public function __construct(?string $path)
    {
        if ($path == null) {
            throw new \InvalidArgumentException("Datei erforderlich!");
        }
        if (!@file_exists($path)) {
            throw new \InvalidArgumentException("Datei existiert nicht!");
        }
        if (!@is_readable($path)) {
            throw new \InvalidArgumentException("Datei ist nicht lesbar!");
        }

        $realpath = realpath($path);
        if (!$realpath) {
            throw new \InvalidArgumentException("{$path} ist kein gültiger Pfad!");
        }
        $this->path = $realpath;
    }

    /**
     * @param $name Name der TeamList
     * @return null|TeamList
     * @throws \InvalidArgumentException
     */
    public function findTeamList(?string $name): ?TeamList
    {
        if ($name == null) {
            throw new \InvalidArgumentException("Name erforderlich!");
        }
        foreach ($this->getTeamLists() as $teamList) {
            if ($teamList->getName() == $name) {
                return $teamList;
            }
        }
        return null;
    }

    /**
     * @return Model\TeamList[]
     */
    public function getTeamLists(): array
    {
        if ($this->teamLists == null) {
            $this->teamLists = [];
            $this->read();
        }
        return $this->teamLists;
    }

    private function read(): void
    {
        try {
            $excel = IOFactory::load($this->path);
        } catch (\Exception $e) {
            throw new \InvalidArgumentException("Datei {$this->path} ist keine gültige Excel-Datei!", 0, $e);
        }
        foreach ($excel->getWorksheetIterator() as $sheet) {
            if ($sheet->getSheetState() == Worksheet::SHEETSTATE_VISIBLE) {
                $teamList = new TeamList($sheet);
                if (!$teamList->isEmpty()) {
                    $this->teamLists[] = $teamList;
                }
            }
        }
    }
}
