<?php
declare(strict_types=1);

namespace Tanzsport\ExcelRankingReader\Model;

use PhpOffice\PhpSpreadsheet\Cell\Cell;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Worksheet\Row;

class Ranking
{

    private string $typ;
    private \stdClass $config;
    private \DateTime $datum;
    private \SplFileInfo $fileInfo;
    private bool $read = false;
    /**
     * @var RankingEntry[]
     */
    private array $eintraege = [];

    public function __construct(string $typ, \stdClass $config, $timestamp, \SplFileInfo $fileInfo)
    {
        $this->typ = $typ;
        $this->config = $config;
        $this->datum = \DateTime::createFromFormat("Ymd H:i:s e", $timestamp . ' 00:00:00 Europe/Berlin');
        $this->fileInfo = $fileInfo;
    }

    private function read(): void
    {
        if (!$this->read) {
            if (strtolower($this->fileInfo->getExtension()) == 'csv') {
                $this->readCsv();
            } else {
                $this->readExcel();
            }
        }
    }

    private function readExcel(): void
    {
        try {
            $excel = IOFactory::load($this->fileInfo->getRealPath());
            $foundFirst = false;
            foreach ($excel->getActiveSheet()->getRowIterator() as $row) {
                /**
                 * @var $row Row
                 */

                $platz = null;
                $partner = null;
                $partnerin = null;
                $verein = null;
                $punkte = null;

                $i = 0;
                $cellIterator = $row->getCellIterator();
                $cellIterator->setIterateOnlyExistingCells(false);
                foreach ($cellIterator as $cell) {
                    /**
                     * @var $cell Cell
                     */
                    if ($cell->getValue() != null && preg_match("/^\d+$/", "{$cell->getValue()}") && !$foundFirst) {
                        $foundFirst = true;
                    }
                    if ($foundFirst) {
                        switch ($i) {
                            case 0:
                                $platz = $cell->getValue();
                                break;
                            case 1:
                                $partner = $cell->getValue();
                                break;
                            case 2:
                                $partnerin = $cell->getValue();
                                break;
                            case 3:
                                $verein = $cell->getValue();
                                break;
                            case 4:
                                $punkte = $cell->getValue();
                                break;
                            default:
                                break;
                        }
                    }
                    $i++;
                }

                if ($foundFirst) {
                    $this->eintraege[] = new RankingEntry($platz, $partner, $partnerin, $verein, $punkte);
                }
            }
            $this->read = true;
        } catch (\Exception $e) {
            throw new \RuntimeException("Daten können nicht gelesen werden!", $e->getCode(), $e);
        }
    }

    private function readCsv(): void
    {
        $fp = @fopen($this->fileInfo->getRealPath(), 'r');
        if ($fp) {
            $foundFirst = false;
            while (($row = fgetcsv($fp, 1000, ";")) !== false) {
                $platz = null;
                $partner = null;
                $partnerin = null;
                $verein = null;
                $punkte = null;

                foreach ($row as $i => $cell) {
                    if (preg_match("/^\d+$/", $cell) && !$foundFirst) {
                        $foundFirst = true;
                    }
                    if ($foundFirst) {
                        switch ($i) {
                            case 0:
                                $platz = $cell;
                                break;
                            case 1:
                                $partner = iconv("ISO-8859-1", "UTF-8", $cell);
                                break;
                            case 2:
                                $partnerin = iconv("ISO-8859-1", "UTF-8", $cell);
                                break;
                            case 3:
                                $verein = iconv("ISO-8859-1", "UTF-8", $cell);
                                break;
                            case 4:
                                $punkte = iconv("ISO-8859-1", "UTF-8", $cell);
                                break;
                            default:
                                break;
                        }
                    }
                }

                if ($foundFirst) {
                    $this->eintraege[] = new RankingEntry($platz, $partner, $partnerin, $verein, $punkte);
                }
            }

            fclose($fp);
            $this->read = true;
        } else {
            throw new \RuntimeException("Datei kann nicht geöffnet werden!");
        }
    }

    /**
     * Gibt den Stand zurück.
     *
     * @return \DateTime
     */
    public function getDatum(): \DateTime
    {
        return $this->datum;
    }

    /**
     * Gibt alle Ranglisteneinträge zurück.
     *
     * @return RankingEntry[]
     */
    public function getEintraege(): array
    {
        if (!$this->read) {
            $this->read();;
        }
        return $this->eintraege;
    }

    /**
     * Gibt den Titel zurück.
     *
     * @return array
     */
    public function getTitel($language): string
    {
        // TODO: replace with str_contains after switching to PHP >= 8.0
        if (preg_match('/de/', $language)) {
            return $this->config->de;
        } else {
            return $this->config->en;
        }
    }

    /**
     * Gibt den Typ zur&uuml;ck.
     *
     * @return string
     */
    public function getTyp(): string
    {
        return $this->typ;
    }

    public function __toString()
    {
        return sprintf('Rangliste[TYP=%1$s,DATUM=%2$s]', $this->typ, $this->datum->format("d.m.Y"));
    }
}
