<?php
namespace Tanzsport\Rangliste;

use Tanzsport\Rangliste\Model\Rangliste;

class RanglistenReader
{

	const FILE_REGEX = "/^\w+\_(\d{6})\.(csv|xls|xlsx)$/i";

	/**
	 * @var array
	 */
	private $config;

	/**
	 * @var array
	 */
	private $ranglisten = array();

	/**
	 * @param $path mixed
	 */
	public function __construct($path)
	{
		if ($path == null) {
			throw new \InvalidArgumentException("Datei/Pfad erforderlich!");
		}

		$configFile = __DIR__ . '/ranglisten.json';
		if (!@is_file($configFile) || !@is_readable($configFile)) {
			throw new \InvalidArgumentException("Konfiguration konnte nicht gelesen werden!");
		}
		$this->config = (array)json_decode(file_get_contents($configFile));

		if (is_array($path)) {
			if (count($path) == 0) {
				throw new \InvalidArgumentException("Pfadangabe enthält keine Elemente!");
			}
			foreach ($path as $p) {
				$this->addPath($p);
			}
		} else {
			$this->addPath($path);
		}

		$this->sortRanglisten();
	}

	private function addPath($path)
	{
		if (!@file_exists($path)) {
			throw new \InvalidArgumentException("Datei/Pfad existiert nicht!");
		}
		if (!@is_readable($path)) {
			throw new \InvalidArgumentException("Datei/Pfad ist nicht lesbar!");
		}
		$path = realpath($path);
		if (is_dir($path)) {
			$this->recurseDirectory($path);
		} else {
			$filename = basename($path);
			if (preg_match(self::FILE_REGEX, $filename)) {
				$this->readFile(new \SplFileInfo($path));
			}
		}
	}

	private function recurseDirectory($path)
	{
		$iter = new \DirectoryIterator($path);
		foreach ($iter as $fileInfo) {
			if ($fileInfo->isDot()) continue;
			if (preg_match(self::FILE_REGEX, $fileInfo->getFilename())) {
				$this->readFile(new \SplFileInfo($fileInfo->getPathname()));
			}
		}
	}

	private function readFile(\SplFileInfo $fileInfo)
	{
		if (strpos($fileInfo->getFilename(), "_") > 0) {
			$typ = strtolower(substr($fileInfo->getFilename(), 0, strpos($fileInfo->getFilename(), '_')));
			if (isset($this->config[$typ])) {
				if (!isset($this->ranglisten[$typ])) {
					$this->ranglisten[$typ] = array();
				}
				$timestamp = preg_replace(self::FILE_REGEX, "$1", $fileInfo->getFilename());
				$this->ranglisten[$typ][] = new Rangliste($typ, $this->config[$typ], $timestamp + 20000000, $fileInfo);
			}
		}
	}

	private function sortRanglisten()
	{
		foreach ($this->ranglisten as $typ => $ranglisten) {
			usort($this->ranglisten[$typ], function (Rangliste $a, Rangliste $b) {
				return $a->getDatum()->format('U') <= $b->getDatum()->format('U') ? 1 : -1;
			});
		}
	}

	/**
	 * Gibt den neusten Stand einer spezifischen Rangliste zur&uuml;ck.
	 *
	 * @param $typ Ranglistentyp
	 * @return Rangliste
	 * @throws \InvalidArgumentException
	 */
	public function getNeuesteRangliste($typ)
	{
		if ($typ == null) {
			throw new \InvalidArgumentException("Typ erforderlich!");
		}
		if (!isset($this->ranglisten[$typ]) || count($this->ranglisten[$typ]) == 0) {
			throw new \InvalidArgumentException("Keine Ranglisten für Typ '{$typ}' vorhanden!");
		}

		return $this->ranglisten[$typ][0];
	}

	/**
	 * Gibt alle alle verf&uuml;gbaren Ranglisten mit neustem Stand zur&uuml;ck.
	 *
	 * @return Rangliste[]
	 */
	public function getNeuesteRanglisten()
	{
		$neuesteRanglisten = array();
		foreach ($this->config as $typ => $ranglistenConfig) {
			if (isset($this->ranglisten[$typ]) && count($this->ranglisten[$typ]) > 0) {
				$neuesteRanglisten[] = $this->getNeuesteRangliste($typ);
			}
		}
		return $neuesteRanglisten;
	}

	/**
	 * Gibt eine spezifische Rangliste mit einem spezifischen Stand zur&uuml;ck.
	 *
	 * @param $typ Ranglistentyp
	 * @param \DateTime $datum
	 * @return Rangliste|null
	 * @throws \InvalidArgumentException
	 */
	public function getRangliste($typ, \DateTime $datum)
	{
		if ($typ == null) {
			throw new \InvalidArgumentException("Typ erforderlich!");
		}
		if (!isset($this->ranglisten[$typ]) || count($this->ranglisten[$typ]) == 0) {
			throw new \InvalidArgumentException("Keine Ranglisten für Typ '{$typ}' vorhanden!");
		}

		foreach ($this->ranglisten[$typ] as $rangliste) {
			if ($rangliste->getDatum()->format('Ymd') == $datum->format('Ymd')) {
				return $rangliste;
			}
		}

		return null;
	}

	/**
	 * Gibt alle Daten eines Ranglistentyps zur&uuml;ck.
	 *
	 * @param $typ Ranglistentyp
	 * @return \DateTime[]
	 * @throws \InvalidArgumentException
	 */
	public function getDaten($typ)
	{
		if ($typ == null) {
			throw new \InvalidArgumentException("Typ erforderlich!");
		}
		if (!isset($this->ranglisten[$typ]) || count($this->ranglisten[$typ]) == 0) {
			throw new \InvalidArgumentException("Keine Ranglisten für Typ '{$typ}' vorhanden!");
		}

		$daten = array();
		foreach ($this->ranglisten[$typ] as $rangliste) {
			$daten[] = $rangliste->getDatum();
		}
		return $daten;
	}
}