<?php
namespace Tanzsport\Rangliste\Model;

class Rangliste
{

	/**
	 * @var string
	 */
	private $typ;
	/**
	 * @var \stdClass
	 */
	private $config;

	/**
	 * @var \DateTime
	 */
	private $datum;

	/**
	 * @var \SplFileInfo
	 */
	private $fileInfo;

	/**
	 * @var bool
	 */
	private $read = false;

	/**
	 * @var RanglisteEintrag[]
	 */
	private $eintraege = array();

	public function __construct($typ, $config, $timestamp, \SplFileInfo $fileInfo)
	{
		if ($typ == null) {
			throw new \InvalidArgumentException("Typ erforderlich!");
		}
		if ($config == null) {
			throw new \InvalidArgumentException("Konfiguration erforderlich!");
		}
		if ($timestamp == null) {
			throw new \InvalidArgumentException("Zeitstempel erforderlich!");
		}

		$this->typ = $typ;
		$this->config = $config;
		$this->datum = \DateTime::createFromFormat("Ymd H:i:s e", $timestamp . ' 00:00:00 Europe/Berlin');
		$this->fileInfo = $fileInfo;
	}

	public function read()
	{
		if (!$this->read) {
			if (strtolower($this->fileInfo->getExtension()) == 'csv') {
				$this->readCsv();
			} else {
				$this->readExcel();
			}
		}
	}

	private function readExcel()
	{
		/**
		 * @var \PHPExcel
		 */
		$excel = null;
		try {
			$excel = \PHPExcel_IOFactory::load($this->fileInfo->getRealPath());
			$foundFirst = false;
			foreach ($excel->getActiveSheet()->getRowIterator() as $row) {
				/**
				 * @var $row \PHPExcel_Worksheet_Row
				 */

				$platz = null;
				$partner = null;
				$partnerin = null;
				$verein = null;
				$punkte = null;

				$i = 0;
				$cellIterator = $row->getCellIterator();
				$cellIterator->setIterateOnlyExistingCells(false);
				foreach ($cellIterator as $cell) {
					/**
					 * @var $cell \PHPExcel_Cell
					 */
					if (preg_match("/^\d+$/", $cell->getValue()) && !$foundFirst) {
						$foundFirst = true;
					}
					if ($foundFirst) {
						switch ($i) {
							case 0:
								$platz = $cell->getValue();
								break;
							case 1:
								$partner = $cell->getValue();
								break;
							case 2:
								$partnerin = $cell->getValue();
								break;
							case 3:
								$verein = $cell->getValue();
								break;
							case 4:
								$punkte = $cell->getValue();
								break;
							default:
								break;
						}
					}
					$i++;
				}

				if ($foundFirst) {
					$this->eintraege[] = new RanglisteEintrag($platz, $partner, $partnerin, $verein, $punkte);
				}
			}
			$this->read = true;
		} catch (\Exception $e) {
			throw new \RuntimeException("Daten können nicht gelesen werden!", $e->getCode(), $e);
		}
	}

	private function readCsv()
	{
		$fp = @fopen($this->fileInfo->getRealPath(), 'r');
		if ($fp) {
			$foundFirst = false;
			while (($row = fgetcsv($fp, 1000, ";")) !== false) {
				$platz = null;
				$partner = null;
				$partnerin = null;
				$verein = null;
				$punkte = null;

				foreach ($row as $i => $cell) {
					if (preg_match("/^\d+$/", $cell) && !$foundFirst) {
						$foundFirst = true;
					}
					if ($foundFirst) {
						switch ($i) {
							case 0:
								$platz = $cell;
								break;
							case 1:
								$partner = iconv("ISO-8859-1", "UTF-8", $cell);
								break;
							case 2:
								$partnerin = iconv("ISO-8859-1", "UTF-8", $cell);
								break;
							case 3:
								$verein = iconv("ISO-8859-1", "UTF-8", $cell);
								break;
							case 4:
								$punkte = iconv("ISO-8859-1", "UTF-8", $cell);
								break;
							default:
								break;
						}
					}
				}

				if ($foundFirst) {
					$this->eintraege[] = new RanglisteEintrag($platz, $partner, $partnerin, $verein, $punkte);
				}
			}

			fclose($fp);
			$this->read = true;
		} else {
			throw new \RuntimeException("Datei kann nicht geöffnet werden!");
		}
	}

	/**
	 * Gibt den Stand zur&uuml;ck.
	 *
	 * @return \DateTime
	 */
	public function getDatum()
	{
		return $this->datum;
	}

	/**
	 * Gibt alle Ranglisteneintr&auml;ge zur&uuml;ck.
	 *
	 * @return RanglisteEintrag[]
	 */
	public function getEintraege()
	{
		if (!$this->read) {
			throw new \RuntimeException("Daten wurden noch nicht eingelesen!");
		}
		return $this->eintraege;
	}

	/**
	 * Gibt den Titel zur&uuml;ck.
	 *
	 * @return array
	 */
	public function getTitel($language)
	{
		if (preg_match("/de/", $language)) {
			return $this->config->de;
		} else {
			return $this->config->en;
		}
	}

	/**
	 * Gibt den Typ zur&uuml;ck.
	 *
	 * @return string
	 */
	public function getTyp()
	{
		return $this->typ;
	}

	public function __toString()
	{
		return sprintf('Rangliste[TYP=%1$s,DATUM=%2$s]', $this->typ, $this->datum->format("d.m.Y"));
	}
}
