<?php
declare(strict_types=1);

namespace Tanzsport\ExcelRankingReader;

use Tanzsport\ExcelRankingReader\Model\Ranking;

class ExcelRankingReader
{

    const FILE_REGEX = "/^\w+\_(\d{6})\.(csv|xls|xlsx)$/i";
    private array $config = [];
    private array $ranglisten = [];

    public function __construct($path)
    {
        if ($path == null) {
            throw new \InvalidArgumentException("Datei/Pfad erforderlich!");
        }

        $configFile = __DIR__ . '/ranglisten.json';
        if (!@is_file($configFile) || !@is_readable($configFile)) {
            throw new \InvalidArgumentException("Konfiguration konnte nicht gelesen werden!");
        }
        $this->config = (array)json_decode(file_get_contents($configFile));

        if (is_array($path)) {
            if (count($path) == 0) {
                throw new \InvalidArgumentException("Pfadangabe enthält keine Elemente!");
            }
            foreach ($path as $p) {
                $this->addPath($p);
            }
        } else {
            $this->addPath($path);
        }

        $this->sortRanglisten();
    }

    private function addPath($path): void
    {
        if (!@file_exists($path)) {
            throw new \InvalidArgumentException("Datei/Pfad existiert nicht!");
        }
        if (!@is_readable($path)) {
            throw new \InvalidArgumentException("Datei/Pfad ist nicht lesbar!");
        }
        $path = realpath($path);
        if (is_dir($path)) {
            $this->recurseDirectory($path);
        } else {
            $filename = basename($path);
            if (preg_match(self::FILE_REGEX, $filename)) {
                $this->readFile(new \SplFileInfo($path));
            }
        }
    }

    private function recurseDirectory($path): void
    {
        $iter = new \DirectoryIterator($path);
        foreach ($iter as $fileInfo) {
            if ($fileInfo->isDot()) continue;
            if (preg_match(self::FILE_REGEX, $fileInfo->getFilename())) {
                $this->readFile(new \SplFileInfo($fileInfo->getPathname()));
            }
        }
    }

    private function readFile(\SplFileInfo $fileInfo): void
    {
        if (strpos($fileInfo->getFilename(), "_") > 0) {
            $typ = strtolower(substr($fileInfo->getFilename(), 0, strpos($fileInfo->getFilename(), '_')));
            if (isset($this->config[$typ])) {
                if (!isset($this->ranglisten[$typ])) {
                    $this->ranglisten[$typ] = array();
                }
                $timestamp = preg_replace(self::FILE_REGEX, "$1", $fileInfo->getFilename());
                $this->ranglisten[$typ][] = new Ranking($typ, $this->config[$typ], $timestamp + 20000000, $fileInfo);
            }
        }
    }

    private function sortRanglisten(): void
    {
        foreach ($this->ranglisten as $typ => $ranglisten) {
            usort($this->ranglisten[$typ], function (Ranking $a, Ranking $b) {
                return $a->getDatum()->format('U') <= $b->getDatum()->format('U') ? 1 : -1;
            });
        }
    }

    /**
     * Gibt den neusten Stand einer spezifischen Rangliste zurück.
     *
     * @param string $typ Ranglistentyp
     * @return Ranking
     * @throws \InvalidArgumentException
     */
    public function getNeuesteRangliste(string $typ): Ranking
    {
        if (!isset($this->ranglisten[$typ]) || count($this->ranglisten[$typ]) == 0) {
            throw new \InvalidArgumentException("Keine Ranglisten für Typ '{$typ}' vorhanden!");
        }

        return $this->ranglisten[$typ][0];
    }

    /**
     * Gibt alle alle verfügbaren Ranglisten mit neustem Stand zurück.
     *
     * @return Ranking[]
     */
    public function getNeuesteRanglisten(): array
    {
        $neuesteRanglisten = array();
        foreach ($this->config as $typ => $ranglistenConfig) {
            if (isset($this->ranglisten[$typ]) && count($this->ranglisten[$typ]) > 0) {
                $neuesteRanglisten[] = $this->getNeuesteRangliste($typ);
            }
        }
        return $neuesteRanglisten;
    }

    /**
     * Gibt eine spezifische Rangliste mit einem spezifischen Stand zurück.
     *
     * @param string $typ Ranglistentyp
     * @param \DateTime $datum
     * @return Ranking|null
     * @throws \InvalidArgumentException
     */
    public function getRangliste(string $typ, \DateTime $datum): ?Ranking
    {
        if (!isset($this->ranglisten[$typ]) || count($this->ranglisten[$typ]) == 0) {
            throw new \InvalidArgumentException("Keine Ranglisten für Typ '{$typ}' vorhanden!");
        }

        foreach ($this->ranglisten[$typ] as $rangliste) {
            if ($rangliste->getDatum()->format('Ymd') == $datum->format('Ymd')) {
                return $rangliste;
            }
        }

        return null;
    }

    /**
     * Gibt alle Daten eines Ranglistentyps zurück.
     *
     * @param string $typ Ranglistentyp
     * @return \DateTime[]
     * @throws \InvalidArgumentException
     */
    public function getDaten(string $typ): array
    {
        if (!isset($this->ranglisten[$typ]) || count($this->ranglisten[$typ]) == 0) {
            throw new \InvalidArgumentException("Keine Ranglisten für Typ '{$typ}' vorhanden!");
        }

        $daten = array();
        foreach ($this->ranglisten[$typ] as $rangliste) {
            $daten[] = $rangliste->getDatum();
        }
        return $daten;
    }
}
