<?php
declare(strict_types=1);

namespace Tanzsport\ExcelEventListReader\Model;

/**
 * Event-Liste
 *
 * @package Tanzsport\ExcelEventListReader\Model
 */
class EventList implements \ArrayAccess, \Countable, \Iterator
{
    private array $events;

    private \DateTime $stand;

    private int $key = 0;

    public function __construct(?array $events = null, ?\DateTime $stand = null)
    {
        if (!$events) {
            $this->events = [];
        } else {
            $this->events = $events;
        }
        if ($stand == null) {
            $this->stand = new \DateTime('now');
        } else {
            $this->stand = $stand;
        }
    }

    #[\ReturnTypeWillChange]
    public function offsetExists($offset)
    {
        return isset($this->events[$offset]);
    }

    #[\ReturnTypeWillChange]
    public function offsetGet($offset)
    {
        return $this->events[$offset];
    }

    #[\ReturnTypeWillChange]
    public function offsetSet($offset, $value)
    {
        if ($offset) {
            $this->events[$offset] = $value;
        } else {
            $this->events[] = $value;
        }
    }

    #[\ReturnTypeWillChange]
    public function offsetUnset($offset)
    {
        if ($offset) {
            unset($this->events[$offset]);
        }
    }

    #[\ReturnTypeWillChange]
    public function count()
    {
        return count($this->events);
    }

    #[\ReturnTypeWillChange]
    public function current()
    {
        return $this->events[$this->key];
    }

    #[\ReturnTypeWillChange]
    public function next()
    {
        $this->key++;
    }

    #[\ReturnTypeWillChange]
    public function key()
    {
        return $this->key;
    }

    #[\ReturnTypeWillChange]
    public function valid()
    {
        return isset($this->events[$this->key]);
    }

    #[\ReturnTypeWillChange]
    public function rewind()
    {
        $this->key = 0;
    }

    public function sort(): self
    {
        usort($this->events, array($this, 'compareEvents'));
        return $this;
    }

    public function onlyCurrent(): EventList
    {
        $current = new EventList(null, $this->getStand());
        $now = new \DateTime('now');
        $u = $now->format('U');
        foreach ($this->events as $e) {
            if ($e->datum->format('U') >= $u) {
                $current[] = $e;
            }
        }
        return $current;
    }

    public function compareEvents(Event $a, Event $b): int
    {
        $u1 = $a->datum->format('U');
        $u2 = $b->datum->format('U');
        if ($u1 == $u2) {
            return strcmp($a->titel, $b->titel);
        } else {
            return $u1 < $u2 ? -1 : 1;
        }
    }

    public function getStand(): \DateTime
    {
        return $this->stand;
    }
}
