<?php
declare(strict_types=1);

namespace Tanzsport\ExcelEventListReader;

use PHPUnit\Framework\TestCase;

class EventlisteExcelReaderTest extends TestCase
{
    private function createReader($file)
    {
        return new ExcelEventListReader($file);
    }

    /**
     * @before
     */
    public function before()
    {
        ini_set('date.timezone', 'UTC');
    }

    /**
     * @test
     */
    public function readsExcelFileInGivenOrder()
    {
        $reader = $this->createReader(__DIR__ . '/Eventliste.xlsx');
        $events = $reader->readEventsInWorksheet(1);

        $this->assertNotNull($events);
        $this->assertNotNull($events->getStand());
        $this->assertEquals(4, count($events));

        $event = $events[0];
        $this->assertEquals(\DateTime::createFromFormat('d.m.y H:i:s', '01.06.13 00:00:00'), $event->datum);
        $this->assertEquals('Rosenblattpokal', $event->titel);
        $this->assertCount(1, $event->links);

        $event = $events[1];
        $this->assertEquals(\DateTime::createFromFormat('d.m.y H:i:s', '01.06.13 00:00:00'), $event->datum);
        $this->assertEquals('Biber-Trophy', $event->titel);
        $this->assertCount(1, $event->links);

        $event = $events[2];
        $this->assertEquals(\DateTime::createFromFormat('d.m.Y H:i:s', '20.12.2099 00:00:00'), $event->datum);
        $this->assertEquals('Weihnachtspokal', $event->titel);
        $this->assertCount(1, $event->links);

        $event = $events[3];
        $this->assertEquals(\DateTime::createFromFormat('d.m.Y H:i:s', '06.12.2099 00:00:00'), $event->datum);
        $this->assertEquals('Nikolauspokal', $event->titel);
        $this->assertCount(2, $event->links);

        $iterated = 0;
        foreach ($events as $e) {
            $iterated++;
        }
        $this->assertEquals(4, $iterated);
    }

    /**
     * @test
     */
    public function sortsEvents()
    {
        $reader = $this->createReader(__DIR__ . '/Eventliste.xlsx');
        $events = $reader->readEventsInWorksheet(1);
        $events->sort();
        $this->assertCount(4, $events);
        $this->assertEquals('Nikolauspokal', $events[2]->titel);
    }

    /**
     * @test
     */
    public function copiesOnlyCurrentEvents()
    {
        $reader = $this->createReader(__DIR__ . '/Eventliste.xlsx');
        $allEvents = $reader->readEventsInWorksheet(1);

        $currentEvents = $allEvents->onlyCurrent();
        $this->assertCount(2, $currentEvents);
    }

    /**
     * @test
     */
    public function nonExcelFile()
    {
        $this->expectException(\RuntimeException::class);
        $this->expectExceptionMessage('keine gültige Excel-Datei');
        $reader = $this->createReader(__FILE__);
        $reader->readEventsInWorksheet(1);
    }

    /**
     * @test
     */
    public function nonExistingSheetProducesEmptyList()
    {
        $reader = $this->createReader(__DIR__ . '/Eventliste.xlsx');
        $allEvents = $reader->readEventsInWorksheet(PHP_INT_MAX);
        $this->assertCount(0, $allEvents);
    }
}
