<?php
declare(strict_types=1);

namespace Tanzsport\ExcelEventListReader;

use \PhpOffice\PhpSpreadsheet\Cell\Cell;
use \PhpOffice\PhpSpreadsheet\IOFactory;
use \PhpOffice\PhpSpreadsheet\Shared\Date;
use \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use \PhpOffice\PhpSpreadsheet\Worksheet\Row;

use Tanzsport\ExcelEventListReader\Model\Event;
use Tanzsport\ExcelEventListReader\Model\EventList;

/**
 * Excel-Reader f&uuml;r Event-Listen
 *
 * @package Tanzsport\ExcelEventListReader
 */
class ExcelEventListReader
{
    private string $path;

    private array $columns = [];

    public function __construct(string $path)
    {
        if ($path == null) {
            throw new \InvalidArgumentException("Datei erforderlich!");
        }
        if (!@file_exists($path)) {
            throw new \InvalidArgumentException("Die Datei {$this->path} existiert nicht!");
        }
        if (!@is_readable($path)) {
            throw new \InvalidArgumentException("Die Datei {$this->path} ist nicht lesbar!");
        }

        $this->path = realpath($path);
    }

    public function readEventsInWorksheet(int $worksheetNumber = 1): EventList
    {
        return $this->read($worksheetNumber);
    }

    private function read(int $worksheetNumber): EventList
    {
        $excel = null;
        try {
            $excel = IOFactory::load($this->path);
        } catch (\Exception $e) {
        }
        if ($excel == null) {
            throw new \RuntimeException("Die {$this->path} ist keine gültige Excel-Datei!");
        }

        $seen = 1;

        $stand = new \DateTime(date('c', filemtime($this->path)));
        foreach ($excel->getWorksheetIterator() as $sheet) {
            if ($sheet->getSheetState() == Worksheet::SHEETSTATE_VISIBLE) {
                if ($seen == $worksheetNumber) {
                    return $this->readSheet($sheet, $stand);
                }
                $seen++;
            }
        }

        return new EventList([], $stand);
    }

    private function readSheet(Worksheet $sheet, \DateTime $stand): EventList
    {
        $events = new EventList([], $stand);
        foreach ($sheet->getRowIterator(1) as $row) {
            /**
             * @var Row $row
             */
            if ($row->getRowIndex() == 1) {
                $this->readHeaderRow($row);
            } else {
                $event = $this->readRow($row);
                if ($event !== null) {
                    $events[] = $event;
                }
            }
        }
        return $events;
    }

    private function readHeaderRow(Row $row): void
    {
        foreach ($row->getCellIterator() as $cell) {
            $value = $cell->getValue();
            if ($value) {
                if ($value == 'Datum') {
                    $value = 'DatumVon';
                }
                $this->columns[$cell->getColumn()] = strtolower(trim($value));
            }
        }
    }

    private function readRow(Row $row): ?Event
    {
        $datumVon = null;
        $datumBis = null;
        $titel = null;
        $ausrichter = null;
        $link1 = array();
        $link2 = array();

        foreach ($row->getCellIterator() as $cell) {
            if (isset($this->columns[$cell->getColumn()])) {
                $column = $this->columns[$cell->getColumn()];
                switch ($column) {
                    case 'datumvon':
                        $datumVon = $this->readDate($cell);
                        break;
                    case 'datumbis':
                        $datumBis = $this->readDate($cell);
                        break;
                    case 'titel':
                        $titel = $cell->getValue();
                        break;
                    case 'ausrichter':
                        $ausrichter = $cell->getValue();
                        break;
                    case 'text1':
                        $link1['text'] = $cell->getValue();
                        break;
                    case 'link1':
                        $link1['url'] = $cell->getValue();
                        break;
                    case 'text2':
                        $link2['text'] = $cell->getValue();
                        break;
                    case 'link2':
                        $link2['url'] = $cell->getValue();
                        break;
                    default:
                        break;
                }
            }
        }

        if ($datumVon && $titel) {
            $event = new Event($datumVon, $titel, $ausrichter, $datumBis);
            if (!empty($link1)) {
                $event->addLink($link1['text'], $link1['url']);
            }
            if (!empty($link2)) {
                $event->addLink($link2['text'], $link2['url']);
            }
            return $event;
        } else {
            return null;
        }
    }

    private function readDate(Cell $cell): ?\DateTime
    {
        if ($cell->getValue() !== null) {
            try {
                if (Date::isDateTime($cell) && intval($cell->getValue()) != 0) {
                    return Date::excelToDateTimeObject($cell->getValue());
                } else {
                    return new \DateTime($cell->getValue());
                }
            } catch (\Exception $e) {
                return null;
            }
        } else {
            return null;
        }
    }
}
