<?php
namespace Tanzsport\Events;

use Tanzsport\Events\Model\Event;
use Tanzsport\Events\Model\EventListe;

/**
 * Excel-Reader f&uuml;r Event-Listen
 *
 * @package Tanzsport\Events
 */
class EventlisteExcelReader
{

	/**
	 * @var string
	 */
	private $path;

	/**
	 * @var array
	 */
	private $columns = [];

	public function __construct($path)
	{
		if ($path == null) {
			throw new \InvalidArgumentException("Datei erforderlich!");
		}
		if (!@file_exists($path)) {
			throw new \InvalidArgumentException("Datei existiert nicht!");
		}
		if (!@is_readable($path)) {
			throw new \InvalidArgumentException("Datei ist nicht lesbar!");
		}

		$this->path = realpath($path);
	}

	public function readEventsInWorksheet($worksheetNumber = 1)
	{
		return $this->read($worksheetNumber);
	}

	private function read($worksheetNumber)
	{
		$excel = null;
		try {
			$excel = \PHPExcel_IOFactory::load($this->path);
		} catch (\Exception $e) {

		}
		if ($excel == null) {
			throw new \InvalidArgumentException("Datei ist keine gültige Excel-Datei!");
		}

		$seen = 1;

		$stand = new \DateTime(date('c', filemtime($this->path)));
		foreach ($excel->getWorksheetIterator() as $sheet) {
			if ($sheet->getSheetState() == \PHPExcel_Worksheet::SHEETSTATE_VISIBLE) {
				if ($seen == $worksheetNumber) {
					return $this->readSheet($sheet, $stand);
				}
				$seen++;
			}
		}

		return new EventListe();
	}

	private function readSheet(\PHPExcel_Worksheet $sheet, \DateTime $stand)
	{
		$events = new EventListe(array(), $stand);
		foreach ($sheet->getRowIterator(1) as $row) {
			/**
			 * @PHPExcel_Worksheet_Row $row
			 */
			if ($row->getRowIndex() == 1) {
				$this->readHeaderRow($row);
			} else {
				$event = $this->readRow($row);
				if ($event !== null) {
					$events[] = $event;
				}
			}
		}
		return $events;
	}

	private function readHeaderRow(\PHPExcel_Worksheet_Row $row)
	{
		foreach ($row->getCellIterator() as $cell) {
			$value = $cell->getValue();
			if ($value) {
				if ($value == 'Datum') {
					$value = 'DatumVon';
				}
				$this->columns[$cell->getColumn()] = strtolower(trim($value));
			}
		}
	}

	private function readRow(\PHPExcel_Worksheet_Row $row)
	{
		$datumVon = null;
		$datumBis = null;
		$titel = null;
		$ausrichter = null;
		$link1 = array();
		$link2 = array();

		/**
		 * @var $cell \PHPExcel_Cell
		 */
		foreach ($row->getCellIterator() as $cell) {
			if (isset($this->columns[$cell->getColumn()])) {
				$column = $this->columns[$cell->getColumn()];
				switch ($column) {
					case 'datumvon':
						$datumVon = $this->readDate($cell);
						break;
					case 'datumbis':
						$datumBis = $this->readDate($cell);
						break;
					case 'titel':
						$titel = $cell->getValue();
						break;
					case 'ausrichter':
						$ausrichter = $cell->getValue();
						break;
					case 'text1':
						$link1['text'] = $cell->getValue();
						break;
					case 'link1':
						$link1['url'] = $cell->getValue();
						break;
					case 'text2':
						$link2['text'] = $cell->getValue();
						break;
					case 'link2':
						$link2['url'] = $cell->getValue();
						break;
					default:
						break;
				}
			}
		}

		if ($datumVon && $titel) {
			$event = new Event($datumVon, $titel, $ausrichter, $datumBis);
			if (!empty($link1)) {
				$event->addLink($link1['text'], $link1['url']);
			}
			if (!empty($link2)) {
				$event->addLink($link2['text'], $link2['url']);
			}
			return $event;
		} else {
			return null;
		}
	}

	private function readDate(\PHPExcel_Cell $cell)
	{
		if ($cell->getValue() !== null) {
			try {
				if (\PHPExcel_Shared_Date::isDateTime($cell) && intval($cell->getValue()) != 0) {
					return \PHPExcel_Shared_Date::ExcelToPHPObject($cell->getValue());
				} else {
					return new \DateTime($cell->getValue());
				}
			} catch (\Exception $e) {
				return null;
			}
		} else {
			return null;
		}
	}
}