<?php
declare(strict_types=1);

namespace Tanzsport\ContaoRankingBundle\Controller\FrontendModule;

use Contao\Config;
use Contao\CoreBundle\Controller\FrontendModule\AbstractFrontendModuleController;
use Contao\CoreBundle\Framework\ContaoFramework;
use Contao\CoreBundle\ServiceAnnotation\FrontendModule;
use Contao\FilesModel;
use Contao\Input;
use Contao\ModuleModel;
use Contao\StringUtil;
use Contao\System;
use Contao\Template;
use Psr\Container\ContainerInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Tanzsport\ExcelRankingReader\ExcelRankingReader;
use Tanzsport\ExcelRankingReader\Model\Ranking;

/**
 * @FrontendModule("td_ranglisten",
 *      template="mod_ranglisten_uebersicht",
 *      category="td",
 *      renderer="forward"
 * )
 */
class RanglistenFrontendModuleController extends AbstractFrontendModuleController
{
    private ContaoFramework $framework;
    private LoggerInterface $contaoErrorLogger;

    public function __construct(ContaoFramework $framework, LoggerInterface $contaoErrorLogger)
    {
        $this->framework = $framework;
        $this->framework->initialize(true);
        $this->contaoErrorLogger = $contaoErrorLogger;
    }

    protected function getResponse(Template $template, ModuleModel $model, Request $request): ?Response
    {
        $parameters = $this->getRequestParameters();
        if ($parameters) {
            $template->setName('mod_ranglisten_anzeige');
            return $this->renderRangliste($parameters->rangliste, $parameters->timestamp ?? null, $template, $model);
        } else {
            return $this->renderUebersicht($template, $model);
        }
    }

    private function createReader(ModuleModel $model): ?ExcelRankingReader
    {
        $paths = $this->getFiles($model, $this->getContainer());
        if (count($paths) > 0) {
            return new ExcelRankingReader($paths);
        } else {
            $this->contaoErrorLogger->warning('Keine Pfade im Ranglisten-Module ermittelt', ['module' => $model->id]);
            return null;
        }
    }

    private function renderUebersicht(Template $template, ModuleModel $model): Response
    {
        $links = [];
        $reader = $this->createReader($model);
        if ($reader) {
            foreach ($reader->getNeuesteRanglisten() as $rangliste) {
                $links[] = $this->generateRanglistenLink($rangliste);
            }
        } else {
            $this->contaoErrorLogger->warning('');
        }
        $template->links = $links;
        return $template->getResponse()->setPrivate();
    }

    private function renderRangliste(string $rangliste, ?string $timestamp, Template $template, ModuleModel $model): Response
    {
        $reader = $this->createReader($model);
        if ($reader) {
            $ranking = $timestamp ? $reader->getRangliste($rangliste, \DateTime::createFromFormat('Ymd', $timestamp)) : $reader->getNeuesteRangliste($rangliste);
            if ($ranking) {
                $language = $this->getPageModel()->language;
                $titel = $ranking->getTitel($language);

                $this->getPageModel()->title .= $titel;

                $this->addHeadlineToTemplate($template, $titel);
                $template->language = $language;
                $template->rangliste = $ranking;
                $template->format = $language == 'de' ? 'd.m.Y' : 'Y-m-d';
                $template->referer = 'javascript:history.go(-1)';
                $template->back = $GLOBALS['TL_LANG']['MSC']['goBack'];
                $template->id = uniqid('mod_ranglisten_anzeige_');

                $links = [];
                foreach ($reader->getDaten($rangliste) as $datum) {
                    $link = new \stdClass;
                    $link->current = $ranking->getDatum() == $datum;
                    $link->url = $this->generateRanglistenUrl($ranking, $datum->format('Ymd'));
                    $link->datum = $datum->format($template->format);
                    $links[] = $link;
                }
                $template->links = $links;
            }
        }
        return $template->getResponse()->setPrivate();
    }

    private function generateRanglistenLink(Ranking $rl)
    {
        switch ($this->getPageModel()->language) {
            case 'de':
                return sprintf('<a href="%1$s">%2$s vom %3$s</a>', $this->generateRanglistenUrl($rl), $rl->getTitel('de'), $rl->getDatum()->format('d.m.Y'));
            default:
                return sprintf('<a href="%1$s">%2$s as of %3$s</a>', $this->generateRanglistenUrl($rl), $rl->getTitel('en'), $rl->getDatum()->format('Y-m-d'));
        }
    }

    private function generateRanglistenUrl(Ranking $rl, $timestamp = null)
    {
        $item = $timestamp == null ? $rl->getTyp() : $rl->getTyp() . "-" . $timestamp;
        return $this->getPageModel()->getFrontendUrl($this->isUseAutoItems() ? "/{$item}" : "/rangliste/" . $item);
    }

    /**
     * @return Config
     */
    private function getConfig()
    {
        return $this->framework->getAdapter(Config::class);
    }

    protected function getContainer(): ContainerInterface
    {
        return $this->framework->getAdapter(System::class)->getContainer();
    }

    /**
     * @return Input
     */
    private function getInput()
    {
        return $this->framework->getAdapter(Input::class);
    }

    private function getAbsoluteFilePath(ContainerInterface $container, FilesModel $file): string
    {
        return $container->getParameter('kernel.project_dir') . '/' . $file->path;
    }

    private function getFiles(ModuleModel $model, ContainerInterface $container): array
    {
        $paths = [];
        if ($model->multiSRC) {
            $files = FilesModel::findMultipleByUuids(StringUtil::deserialize($model->multiSRC, true));
            if ($files && $files->count() > 0) {
                while ($files->next()) {
                    $absolute = $this->getAbsoluteFilePath($container, $files->current());
                    if ((is_file($absolute) || is_dir($absolute)) && file_exists($absolute)) {
                        $paths[] = $absolute;
                    } else {
                        $this->contaoErrorLogger->warning("Pfad {$absolute} sollte eine Ranglisten-Datei oder ein Pfad sein, ist aber ungültig");
                    }
                }
            }
        }
        return $paths;
    }

    private function getRequestParameters(): ?\stdClass
    {
        $requested = $this->isUseAutoItems() ? $this->getInput()->get('auto_item') : $this->getInput->get('rangliste');
        if ($requested) {
            $parts = explode('-', $requested);
            return count($parts) > 1 ? (object)['rangliste' => $parts[0], 'timestamp' => $parts[1]] : (object)['rangliste' => $parts[0]];
        } else {
            return null;
        }
    }

    private function isUseAutoItems(): bool
    {
        return (bool)$this->getConfig()->get('useAutoItem');
    }

}
