<?php

namespace Tanzsport;

/**
 * Class LigaMannschaftModel
 * @package Tanzsport
 *
 * @property LigaModel $liga Zugeh&ouml;hrige Liga
 * @property bool $saisonAbgelaufen Gibt an, ob die Saison bereits abgelaufen ist
 */
class LigaMannschaftModel extends \Model
{

    const TABLE = "tl_td_erg_liga_mannschaft";

    protected static $strTable = self::TABLE;

    private $_kontakteAsArray;
    private $_trainerAsArray;

    public static function findByLiga($intLigaId)
    {
        if (!$intLigaId) {
            throw new \InvalidArgumentException("Liga-ID erforderlich!");
        }
        return static::findBy('pid', $intLigaId, array('order' => 'verein_name, team_name'));
    }

    /**
     * @param $intLigaId
     * @return \Model\Collection|null|static
     */
    public static function findPublishedByLigaOrderByStartnr($intLigaId)
    {
        if (!$intLigaId) {
            throw new \InvalidArgumentException("Liga-ID erforderlich!");
        }
        $t = self::getTable();
        return static::findBy(array("$t.pid = ?", "$t.published = ?"), array($intLigaId, 1), array('order' => "$t.startnr"));
    }

    public function __get($key)
    {
        switch ($key) {
            case 'kapitaen':
                return $this->getKapitaen();
            case 'kontakteAsArray':
                return $this->getKontakteAsArray();
            case 'liga':
                return $this->getRelated('pid');
            case 'team':
                return $this->getTeam();
            case 'trainerAsArray':
                return $this->getTrainerAsArray();
            case 'verein';
                return $this->getVerein();
            default:
                return parent::__get($key);
        }
    }

    public function __isset($key)
    {
        switch ($key) {
            case 'kapitaen':
                return $this->getKapitaen() != null;
            case 'kontakteAsArray':
                return $this->getKontakteAsArray() != null;
            case 'liga':
                return $this->liga != null;
            case 'team':
                return $this->getTeam() != null;
            case 'trainerAsArray':
                return $this->getTrainerAsArray() != null;
            case 'verein':
                return $this->getVerein() != null;
            default:
                return parent::__isset($key);
        }
    }

    public function __set($key, $value)
    {
        switch ($key) {
            case 'liga':
                parent::__set('pid', $value->id);
                break;
            default:
                parent::__set($key, $value);
        }
    }

    /**
     * @param $pid
     * @return LigaMannschaftModel
     */
    public function createCopyFor($pid)
    {
        $copy = new LigaMannschaftModel();
        $copy->pid = $pid;
        $copy->tstamp = date('U');
        $copy->startnr = $this->startnr;
        $copy->verein_name = $this->verein_name;
        $copy->verein_url = $this->verein_url;
        $copy->team_name = $this->team_name;
        $copy->team_url = $this->team_url;
        $copy->trainer = $this->trainer;
        $copy->kapitaen_name = $this->kapitaen_name;
        $copy->kapitaen_email = $this->kapitaen_email;
        $copy->kontakt = $this->kontakt;
        $copy->bemerkungen = $this->bemerkungen;
        $copy->published = '0';
        $copy->aufstieg = null;
        $copy->platz = null;

        return $copy;
    }

    private function getKapitaen()
    {
        if ($this->kapitaen_name) {
            if (!$this->liga->saisonAbgelaufen) {
                return $this->emailOrText($this->kapitaen_email, $this->kapitaen_name);
            } else {
                return $this->kapitaen_name;
            }
        }
    }

    public function asArray()
    {
        $thisAsArray = $this->row();
        foreach (['kontakt', 'trainer'] as $property) {
            $thisAsArray[$property] = deserialize($thisAsArray[$property], true);
        }
        return $thisAsArray;
    }

    private function getKontakteAsArray()
    {
        if ($this->_kontakteAsArray == null) {
            $kontakte = deserialize($this->kontakt, true);
            $this->_kontakteAsArray = array();
            foreach ($kontakte as $i => $k) {
                if (!$this->liga->saisonAbgelaufen) {
                    $this->_kontakteAsArray[] = $this->emailOrText($k['email'], $k['name']);
                } else {
                    $this->_kontakteAsArray[] = $k['name'];
                }
            }
        }
        return $this->_kontakteAsArray;
    }

    private function getTeam()
    {
        if ($this->team_url) {
            if (!$this->liga->saisonAbgelaufen) {
                return sprintf('<a href="%1$s" target="_blank">%2$s</a>', $this->team_url, $this->team_name);
            } else {
                return $this->team_name;
            }
        } else {
            return $this->team_name;
        }
    }

    private function getTrainerAsArray()
    {
        if ($this->_trainerAsArray == null) {
            $trainer = deserialize($this->trainer, true);
            $this->_trainerAsArray = array();
            foreach ($trainer as $i => $t) {
                if (!$this->liga->saisonAbgelaufen) {
                    $this->_trainerAsArray[] = $this->emailOrText($t['email'], $t['name']);
                } else {
                    $this->_trainerAsArray[] = $t['name'];
                }
            }
        }
        return $this->_trainerAsArray;
    }

    private function getVerein()
    {
        if ($this->verein_url) {
            if (!$this->liga->saisonAbgelaufen) {
                return sprintf('<a href="%1$s" target="_blank">%2$s</a>', $this->verein_url, $this->verein_name);
            } else {
                return $this->verein_name;
            }
        } else {
            return $this->verein_name;
        }
    }

    private function emailOrText($strEmail, $strText)
    {
        if ($strEmail) {
            return sprintf('<a href="{{email_url::mailto:%1$s}}">%2$s</a>', $strEmail, $strText);
        } else {
            return $strText;
        }
    }

}
