<?php
declare(strict_types=1);

namespace Tanzsport\ContaoFormationenBundle\Controller\Api;

use Contao\ModuleModel;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\Routing\Annotation\Route;
use Tanzsport\Ergebnisarchiv;

class MannschaftsdatenMeldenController extends AbstractApiController
{
    use LoadLigenTrait, LoadMannschaftenTrait, EmailTrait;

    /**
     * @Route("/_api/formationen/mannschaftsdaten/{module}/ligen", name="td.formationen.mannschaftsdaten.ligen")
     */
    public function ligen(string $module): Response
    {
        $model = ModuleModel::findById($module);
        if (!$model) {
            throw $this->createNotFoundException("Modul {$module} existiert nicht!");
        }
        try {
            return $this->json($this->loadLigen($model))->setPrivate();
        } catch (\Throwable $e) {
            $this->contaoErrorLogger->error("Fehler bei der Ermittlung der Ligen: {$e->getMessage()}", ['controller' => self::class, 'exception' => $e]);
            throw $e;
        }
    }

    /**
     * @Route("/_api/formationen/mannschaftsdaten/mannschaften", name="td.formationen.mannschaftsdaten.mannschaften")
     */
    public function mannschaften(Request $request): Response
    {
        if (!$request->query->get('l')) {
            throw new BadRequestHttpException('Parameter Liga ist erforderlich!');
        }

        try {
            return $this->json($this->loadMannschaften($request->query->get('l')))->setPrivate();
        } catch (\Throwable $e) {
            $this->contaoErrorLogger->error("Fehler bei der Ermittlung der Mannschaften: {$e->getMessage()}", ['controller' => self::class, 'exception' => $e]);
            throw $e;
        }
    }

    /**
     * @Route("/_api/formationen/mannschaftsdaten/melden", name="td.formationen.mannschaftsdaten.melden", methods={"POST"})
     */
    public function melden(Request $request): Response
    {
        try {
            $data = json_decode($request->getContent(), false);
            $sent = $this->mailen($data, 'melden_mannschaftsdaten');
            if ($sent) {
                return (new Response('Turnierdaten übermittelt', 200))->setPrivate();
            } else {
                $this->contaoErrorLogger->error('Fehler beim E-Mailversand', ['controller' => self::class]);
                return (new Response('Turnierdaten konnten nicht übermittelt werden', 500))->setPrivate();
            }
        } catch (\Throwable $e) {
            $this->contaoErrorLogger->error("Fehler bei der Übermittlung der Mannschaftsdaten: {$e->getMessage()}", ['controller' => self::class, 'exception' => $e]);
            throw $e;
        }
    }

    protected function buildSubject(\stdClass $data): string
    {
        $subject = '[';
        if ($data->liga->turnierart == Ergebnisarchiv::TURNIERART_JMD) {
            $subject .= 'jmd.tanzsport.de';
        } else {
            $subject .= 'formationen.de';
        }
        $subject .= '] Teamdaten ';
        $subject .= $data->liga->titel;
        $subject .= ' | ';
        $subject .= $data->mannschaft->verein_name;
        $subject .= ' ';
        $subject .= $data->mannschaft->team_name;
        return $subject;
    }
}
