<?php

namespace Tanzsport;

/**
 * Class LigaModel
 * @package Tanzsport
 *
 * @property string $bereich Liga-Bereich
 * @property string $bezeichnung Zusatzbezeichnung
 * @property string $liga Liga
 * @property string $nr Nr
 * @property bool $published ver&ouml;ffentlicht ja/nein
 * @property string $turnierart Turnierart
 * @property SaisonModel $saison Zugeh&ouml;hrige Saison
 * @property bool $saisonAbgelaufen Gibt an, ob die Saison bereits abgelaufen ist
 * @property string $saisonTitel Saison-Titel
 * @property string $titel Liga-Titel
 */
class LigaModel extends \Model
{

    protected static $strTable = "tl_td_erg_liga";

    public static function findOnePublishedByJahrBereichLigaTurnierart($intJahr, $strBereich, $strLiga, $strTurnierart, $intNr = null)
    {
        $objSaison = SaisonModel::findOneBy('jahr', $intJahr);
        if ($objSaison != null) {
            $l = self::getTable();
            $arrColumns = array("$l.published = ?", "$l.pid = ?", "$l.bereich = ?", "$l.liga = ?", "$l.turnierart = ?");
            $arrValues = array(1, $objSaison->id, $strBereich, $strLiga, $strTurnierart);
            if ($intNr) {
                $arrColumns[] = "$l.nr = ?";
                $arrValues[] = $intNr;
            }
            return self::findOneBy($arrColumns, $arrValues);
        }
    }

    /**
     * @param SaisonModel $saison
     * @param array $turnierarten
     * @return \Model\Collection|null|static
     */
    public static function findBySaison(SaisonModel $saison, array $turnierarten = null)
    {
        $l = self::getTable();
        $arrColumns = ["$l.published = ?", "$l.pid = ?"];
        $arrValues = [1, $saison->id];
        if ($turnierarten != null && count($turnierarten) > 0) {
            $inQuery = "$l.turnierart IN (";
            foreach ($turnierarten as $i => $turnierart) {
                if ($i <> 0) {
                    $inQuery .= ',';
                }
                $inQuery .= '?';
                $arrValues[] = $turnierart;
            }
            $inQuery .= ')';
            $arrColumns[] = $inQuery;
        }
        return self::findBy($arrColumns, $arrValues, ['order' => 'bereichOrder, ligaOrder, turnierartOrder, nr, bezeichnung']);
    }

    public function __get($key)
    {
        switch ($key) {
            case 'bereichTitel':
                return $this->getBereichTitel();
            case 'saison':
                return $this->getRelated('pid');
            case 'saisonAbgelaufen':
                return $this->isSaisonAbgelaufen();
            case 'saisonTitel':
                return $this->saison->getSaisonTitel($this->turnierart);
            case 'titel':
                return self::getTitel($this);
            default:
                return parent::__get($key);
        }
    }

    public function __isset($key)
    {
        switch ($key) {
            case 'bereichTitel':
            case 'saisonAbgelaufen':
            case 'saisonTitel':
            case 'titel':
                return true;
            case 'saison':
                return $this->saison != null;
            default:
                return parent::__isset($key);
        }
    }

    /**
     * @param $pid
     * @return LigaModel
     */
    public function createCopyFor($pid)
    {
        $copy = new LigaModel();
        $copy->pid = $pid;
        $copy->tstamp = date('U');
        $copy->bereich = $this->bereich;
        $copy->bereichOrder = $this->bereichOrder;
        $copy->liga = $this->liga;
        $copy->ligaOrder = $this->ligaOrder;
        $copy->turnierart = $this->turnierart;
        $copy->turnierartOrder = $this->turnierartOrder;
        $copy->nr = $this->nr;
        $copy->bezeichnung = $this->bezeichnung;
        $copy->published = '0';
        $copy->text = $this->text;

        return $copy;
    }

    private function getBereichTitel()
    {
        \System::loadLanguageFile('tl_td_erg');
        return $GLOBALS['TL_LANG'][\Tanzsport\Ergebnisarchiv::LNG_KEY]['public'][$this->bereich];
    }

    public static function getTitel($obj)
    {
        \System::loadLanguageFile('tl_td_erg');

        $titel = $GLOBALS['TL_LANG'][\Tanzsport\Ergebnisarchiv::LNG_KEY][$obj->liga];
        if (!in_array($obj->liga, array(Ergebnisarchiv::LIGA_1BL, Ergebnisarchiv::LIGA_2BL))) {
            $titel .= " ";
            $titel .= $GLOBALS['TL_LANG'][\Tanzsport\Ergebnisarchiv::LNG_KEY]['short'][$obj->bereich];
        }
        if ($obj->nr > 0) {
            $titel .= " ";
            if ($obj->bezeichnung) {
                $titel .= $obj->bezeichnung;
            } else {
                $titel .= $obj->nr;
            }
        }
        $titel .= " ";
        $titel .= $GLOBALS['TL_LANG'][\Tanzsport\Ergebnisarchiv::LNG_KEY][$obj->turnierart];
        if ($obj->text) {
            $titel .= " (";
            $titel .= $obj->text;
            $titel .= ")";
        }

        return $titel;
    }

    private function isSaisonAbgelaufen()
    {
        if (isset($GLOBALS['TL_CONFIG']['td_erg_ausblenden']) && $GLOBALS['TL_CONFIG']['td_erg_ausblenden']) {
            $objSaison = $this->saison;
            if ($objSaison) {
                $intJahr = $objSaison->jahr;
                if ($intJahr) {
                    $objLimit = null;
                    if ($this->turnierart == Ergebnisarchiv::TURNIERART_JMD) {
                        $intJahr++;
                        $objLimit = new \DateTime("{$intJahr}-01-01");
                    } else {
                        $objLimit = new \DateTime("{$intJahr}-10-01");
                    }
                    $objNow = new \DateTime('now');
                    return $objNow >= $objLimit;
                }
            }
            return true;
        }

        return false;
    }

    /**
     * @param SaisonModel $objSaison
     * @param $strBereich
     * @return \Model\Collection|null|static
     */
    public static function findPublishedBySaisonAndLigabereich(SaisonModel $objSaison, $strBereich)
    {
        $t = self::$strTable;
        return self::findBy(["$t.published = ?", "$t.pid = ?", "$t.bereich = ?"], [1, $objSaison->id, $strBereich], ['order' => "$t.turnierartOrder, $t.ligaOrder, $t.nr"]);
    }
}
