<?php

namespace Tanzsport;

/**
 * Class FreiesturnierModel
 * @package Tanzsport
 */
class FreiesturnierModel extends \Model
{

    protected static $strTable = "tl_td_erg_turnier";

    /**
     * @param SaisonModel $saison
     * @param array $turnierarten
     * @return \Model\Collection|null|static
     */
    public static function findBySaison(SaisonModel $saison, array $turnierarten = null)
    {
        $t = self::getTable();
        $arrColumns = ["$t.turnier_published = ?", "$t.pid = ?"];
        $arrValues = [1, $saison->id];
        if ($turnierarten != null && count($turnierarten) > 0) {
            $inQuery = "$t.turnierart IN (";
            foreach ($turnierarten as $i => $turnierart) {
                if ($i <> 0) {
                    $inQuery .= ',';
                }
                $inQuery .= '?';
                $arrValues[] = $turnierart;
            }
            $inQuery .= ')';
            $arrColumns[] = $inQuery;
        }
        return self::findBy($arrColumns, $arrValues, ['order' => 'bereichOrder, turnierartOrder, titel']);
    }

    public function __get($key)
    {
        switch ($key) {
            case 'bereichTitel':
                return $this->getBereichTitel();
            case 'saison':
                return $this->getRelated('pid');
            case 'saisonAbgelaufen':
                return $this->isSaisonAbgelaufen();
            case 'saisonTitel':
                return $this->saison->getSaisonTitel($this->turnierart);
            case 'datumAsObject':
                $dt = \DateTime::createFromFormat('U', $this->datum);
                if (!$dt) {
                    $dt = \DateTime::createFromFormat('U', 0);
                }
                return $dt->setTimezone(new \DateTimeZone('Europe/Berlin'));
            case 'hasAnfahrt':
                return $this->hasAnfahrt();
            case 'hasKarten':
                return $this->hasKarten();
            case 'hasTurnierstaette':
                return $this->hasTurnierstaette();
            case 'hasVorverkauf':
                return $this->hasVorverkauf();
            case 'hasZeiten':
                return $this->hasZeiten();
            case 'karten':
                return deserialize(parent::__get('karten'));
            case 'zeiten_beginn':
                return deserialize(parent::__get('zeiten_beginn'));
            default:
                return parent::__get($key);
        }
    }

    public function __isset($key)
    {
        switch ($key) {
            case 'bereichTitel':
            case 'saisonAbgelaufen':
            case 'saisonTitel':
            case 'saison':
                return $this->saison != null;
            case 'datumAsObject':
                return isset($this->datum);
            case 'hasAnfahrt':
                return true;
            case 'hasKarten':
                return true;
            case 'hasTurnierstaette':
                return true;
            case 'hasVorverkauf':
                return true;
            case 'hasZeiten':
                return true;
            case 'karten':
                return parent::__isset('karten');
            case 'zeiten_beginn':
                return parent::__isset('zeiten_beginn');
            default:
                return parent::__isset($key);
        }
    }

    public function asArray()
    {
        $thisAsArray = $this->row();
        foreach (['zeiten_beginn', 'karten'] as $property) {
            $thisAsArray[$property] = deserialize($thisAsArray[$property], true);
        }
        $thisAsArray['datumText'] = $this->datumAsObject->format('d.m.Y');
        return $thisAsArray;
    }

    private function getBereichTitel()
    {
        \System::loadLanguageFile('tl_td_erg');
        return $GLOBALS['TL_LANG'][\Tanzsport\Ergebnisarchiv::LNG_KEY]['public'][$this->bereich];
    }

    private function isSaisonAbgelaufen()
    {
        if (isset($GLOBALS['TL_CONFIG']['td_erg_ausblenden']) && $GLOBALS['TL_CONFIG']['td_erg_ausblenden']) {
            $objSaison = $this->saison;
            if ($objSaison) {
                $intJahr = $objSaison->jahr;
                if ($intJahr) {
                    $objLimit = null;
                    if ($this->turnierart == Ergebnisarchiv::TURNIERART_JMD) {
                        $intJahr++;
                        $objLimit = new \DateTime("{$intJahr}-01-01");
                    } else {
                        $objLimit = new \DateTime("{$intJahr}-10-01");
                    }
                    $objNow = new \DateTime('now');
                    return $objNow >= $objLimit;
                }
            }
            return true;
        }

        return false;
    }

    private function hasAnfahrt()
    {
        return $this->anfahrt_text || $this->anfahrt_url;
    }

    private function hasTurnierstaette()
    {
        return $this->tst_halle || $this->tst_telefon || $this->tst_adresse;
    }

    private function hasKarten()
    {
        return !empty($this->karten);
    }

    private function hasVorverkauf()
    {
        return $this->vvk_text || $this->vvk_anschrift || $this->vvk_telefon || $this->vvk_fax || $this->vvk_email_adresse || $this->vvk_online_url || $this->vvk_bemerkungen;
    }

    private function hasZeiten()
    {
        return !empty($this->zeiten_beginn) || $this->zeiten_einlass;
    }

    public static function findPublishedBySaisonAndLigabereich(SaisonModel $objSaison, $strBereich)
    {
        $t = self::$strTable;
        return self::findBy(["$t.turnier_published = ?", "$t.pid = ?", "$t.bereich = ?"], [1, $objSaison->id, $strBereich], ['order' => "$t.turnierartOrder, $t.datum, $t.titel"]);
    }
}
