<?php
declare(strict_types=1);

namespace Tanzsport\ContaoFormationenBundle\Controller\Api;

use Contao\ModuleModel;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\Routing\Annotation\Route;
use Tanzsport\Ergebnisarchiv;

class ErgebnisMeldenController extends AbstractApiController
{
    use LoadLigenTrait, StammdatenErgebnisTrait, EmailTrait;

    /**
     * @Route("/_api/formationen/ergebnis/{module}/ligen", name="td.formationen.ergebnis.ligen")
     */
    public function ligen(string $module): Response
    {
        $model = ModuleModel::findById($module);
        if (!$model) {
            throw $this->createNotFoundException("Modul {$module} existiert nicht!");
        }
        try {
            return $this->json($this->loadLigen($model))->setPrivate();
        } catch (\Throwable $e) {
            $this->contaoErrorLogger->error("Fehler bei der Ermittlung der Ligen: {$e->getMessage()}", ['controller' => self::class, 'exception' => $e]);
            throw $e;
        }
    }

    /**
     * @Route("/_api/formationen/ergebnis/stammdaten-liga", name="td.formationen.ergebnis.stammdaten-liga")
     */
    public function stammdatenLiga(Request $request): Response
    {
        if (!$request->query->get('l')) {
            throw new BadRequestHttpException('Parameter Liga ist erforderlich!');
        }

        try {
            return $this->json($this->loadStammdatenLiga($request->query->get('l')))->setPrivate();
        } catch (\Throwable $e) {
            $this->contaoErrorLogger->error("Fehler bei der Ermittlung der Liga-Stammdaten: {$e->getMessage()}", ['controller' => self::class, 'exception' => $e]);
            throw $e;
        }
    }

    /**
     * @Route("/_api/formationen/ergebnis/stammdaten-turnier", name="td.formationen.ergebnis.stammdaten-turnier")
     */
    public function stammdatenTurnier(Request $request): Response
    {
        if (!$request->query->get('t')) {
            throw new BadRequestHttpException('Parameter Turnier ist erforderlich!');
        }

        try {
            return $this->json($this->loadStammdatenTurnier($request->query->get('t')))->setPrivate();
        } catch (\Throwable $e) {
            $this->contaoErrorLogger->error("Fehler bei der Ermittlung der Turnier-Stammdaten: {$e->getMessage()}", ['controller' => self::class, 'exception' => $e]);
            throw $e;
        }
    }

    /**
     * @Route("/_api/formationen/ergebnis/melden", name="td.formationen.ergebnis.melden", methods={"POST"})
     */
    public function melden(Request $request): Response
    {
        try {
            $data = json_decode($request->getContent(), false);
            $sent = $this->mailen($data, 'melden_ergebnis');
            if ($sent) {
                return (new Response('Ergebnis übermittelt', 200))->setPrivate();
            } else {
                $this->contaoErrorLogger->error('Fehler beim E-Mailversand', ['controller' => self::class]);
                return (new Response('Ergebnis konnte nicht übermittelt werden', 500))->setPrivate();
            }
        } catch (\Throwable $e) {
            $this->contaoErrorLogger->error("Fehler bei der Übermittlung des Ergebnis: {$e->getMessage()}", ['controller' => self::class, 'exception' => $e]);
            throw $e;
        }
    }

    protected function buildSubject(\stdClass $data): string
    {
        $subject = '[';
        if ($data->liga->turnierart == Ergebnisarchiv::TURNIERART_JMD) {
            $subject .= 'jmd.tanzsport.de';
        } else {
            $subject .= 'formationen.de';
        }
        $subject .= '] Ergebnismeldung ';
        $subject .= $data->liga->titel;
        $subject .= ' | ';
        $subject .= $data->turnier->datumText;
        $subject .= ' | ';
        $subject .= $data->turnier->ort;
        return $subject;
    }
}
