<?php
declare(strict_types=1);

namespace Tanzsport\ContaoDtvApiBundle\Controller\FrontendModule;

use Contao\CoreBundle\Controller\FrontendModule\AbstractFrontendModuleController;
use Contao\CoreBundle\ServiceAnnotation\FrontendModule;
use Contao\FrontendTemplate;
use Contao\ModuleModel;
use Contao\Template;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Tanzsport\API\Verein\VereinsDatenbankService;
use Tanzsport\ContaoDtvApiBundle\UI\FormPagination;

/**
 * @FrontendModule("td_vereinssuche",
 *     category="td",
 *     template="mod_vereinssuche_formular",
 *     renderer="forward"
 * )
 */
class VereinsSucheFrontendModuleController extends AbstractFrontendModuleController
{
    private static string $formId = 'mod_vereinssuche_formular';

    private VereinsDatenbankService $service;
    private LoggerInterface $contaoErrorLogger;

    public function __construct(VereinsDatenbankService $service, LoggerInterface $contaoErrorLogger)
    {
        $this->service = $service;
        $this->contaoErrorLogger = $contaoErrorLogger;
    }

    protected function getResponse(Template $template, ModuleModel $model, Request $request): ?Response
    {
        try {
            if ($request->isMethod('POST') && $request->request->get('FORM_SUBMIT') === self::$formId) {
                $params = [];
                $keys = ['name', 'landesverband', 'standort', 'seite'];
                foreach ($keys as $key) {
                    if ($request->request->has($key)) {
                        $params[$key] = $request->request->get($key);
                    }
                }
                $result = $this->findVereine($params);

                $template = new FrontendTemplate('mod_vereinssuche_ergebnis');
                $template->ergebnis = $result;
                $template->pager = new FormPagination($result['anzahlGesamt'], 10, 7, 'seite', true);
                return $template->getResponse()->setPrivate();
            } else {
                $GLOBALS['TL_JAVASCRIPT'][] = 'bundles/contaodtvapi/typeahead/0.10.2/typeahead.bundle.min.js|static';
                $GLOBALS['TL_JAVASCRIPT'][] = 'bundles/contaodtvapi/vdb/standorte.js|static';

                $template->formId = self::$formId;
                $template->remote = "/_api/proxy/vereine?q=s&p=%QUERY";
                $template->landesverbaende = $this->loadLandesverbaende();
                return $template->getResponse()->setPrivate();
            }
        } catch (\Throwable $e) {
            $this->contaoErrorLogger->log('error', 'Error querying Vereinsdatenbank', ['exception' => $e]);
            $template = new FrontendTemplate('mod_api_error');
            $template->message = $e->getMessage();
            return $template->getResponse()->setPrivate();
        }
    }

    private function loadLandesverbaende()
    {
        return $this->service->getLandesverbaende();
    }

    private function findVereine(array $params)
    {
        $name = isset($params['name']) ? $params['name'] : null;
        $landesverbaende = isset($params['landesverband']) ? $params['landesverband'] : null;
        $standort = isset($params['standort']) ? $params['standort'] : null;
        $seite = isset($params['seite']) ? $params['seite'] : 0;
        return $this->service->findVereine($name, $landesverbaende, $standort, $seite);
    }
}
