<?php
declare(strict_types=1);

namespace Tanzsport\ContaoDtvApiBundle\Controller\FrontendModule;

use Contao\CoreBundle\Controller\FrontendModule\AbstractFrontendModuleController;
use Contao\CoreBundle\ServiceAnnotation\FrontendModule;
use Contao\FrontendTemplate;
use Contao\ModuleModel;
use Contao\Template;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Tanzsport\API\Turnier\FormationswettbewerbDatenbankService;
use Tanzsport\API\Turnier\FormationswettbewerbSuche;
use Tanzsport\API\Turnier\Suchergebnis;
use Tanzsport\ContaoDtvApiBundle\Controller\Pdf\TurnierSuchePdfExportController;
use Tanzsport\ContaoDtvApiBundle\UI\FormPagination;
use Tanzsport\Validation\BindingResult;

/**
 * @FrontendModule("td_turnierformationsuche",
 *     category="td",
 *     template="mod_turnierformationsuche_formular",
 *     renderer="forward"
 * )
 */
class TurnierFormationSucheFrontendModuleController extends AbstractFrontendModuleController
{
    use TurnierSucheTrait;

    private static string $formId = 'mod_turnierformationsuche_formular';

    private FormationswettbewerbDatenbankService $service;
    private LoggerInterface $contaoErrorLogger;

    public function __construct(FormationswettbewerbDatenbankService $service, LoggerInterface $contaoErrorLogger)
    {
        $this->service = $service;
        $this->contaoErrorLogger = $contaoErrorLogger;
    }

    protected function getResponse(Template $template, ModuleModel $model, Request $request): ?Response
    {
        try {
            if ($request->isMethod('POST') && $request->request->get('FORM_SUBMIT') === self::$formId) {
                if ($request->request->get('action') === 'reset') {
                    $this->service->reset();
                    return $this->createResponseForm($template);
                } else {
                    if ($request->request->has('search')) {
                        $search = $this->service->unserialize($request->request->get('search'));
                    } else {
                        $search = $this->bindSearchParameters($request);
                    }
                    $bindingResult = $this->service->validate($search);
                    if (!$bindingResult->isError()) {
                        return $this->createResponseResult($model, $search, $this->service->suchen($search));
                    } else {
                        return $this->createResponseForm($template, $bindingResult, $search);
                    }
                }
            } else {
                return $this->createResponseForm($template);
            }
        } catch (\Throwable $e) {
            $this->contaoErrorLogger->log('error', 'Error querying FormationswettbewerbDatenbank', ['exception' => $e]);
            $template = new FrontendTemplate('mod_api_error');
            $template->message = $e->getMessage();
            return $template->getResponse()->setPrivate();
        }
    }

    private function createResponseForm(Template $template, ?BindingResult $bindingResult = null, ?FormationswettbewerbSuche $search = null): ?Response
    {
        $GLOBALS['TL_JAVASCRIPT'][] = 'bundles/contaodtvapi/tdb/common.js|static';
        $GLOBALS['TL_JAVASCRIPT'][] = 'bundles/contaodtvapi/tdb/formation.js|static';

        if (!$bindingResult) {
            $bindingResult = BindingResult::createEmpty();
        }
        if (!$search) {
            $search = $this->createDefaultSearchParameters();
        }

        $template->formId = self::$formId;
        $template->exportUrl = $this->generateUrl(TurnierSuchePdfExportController::class, ['service' => 'formation']);
        $template->binding = $bindingResult;
        $template->suche = $search;
        $template->landesverbaende = $this->service->getLandesverbaende();
        $template->ligen = $this->service->getLigen();
        if ($search->getLiga()) {
            $template->turnierarten = $this->service->getTurnierarten($search->getLiga());
        }
        $template->remote = "/_api/proxy/turnier-formation";
        return $template->getResponse()->setPrivate();
    }

    private function createResponseResult(ModuleModel $model, FormationswettbewerbSuche $search, Suchergebnis $result): ?Response
    {
        $template = new FrontendTemplate('mod_turniersuche_ergebnis');
        $template->formId = self::$formId;
        $template->exportUrl = $this->generateUrl(TurnierSuchePdfExportController::class, ['service' => 'formation']);
        $template->suche = $search;
        $template->ergebnis = $result;
        $template->serialized = $this->service->serialize($search);
        $template->unserialized = $this->service->unserialize($template->serialized);
        $template->pager = new FormPagination($result->getAnzahlGesamt(), 10, 7, 'seite', true);
        $template->vereinsLinkBase = $this->generateVereinsLinkBase($model);
        return $template->getResponse()->setPrivate();
    }

    private function bindSearchParameters(Request $request): FormationswettbewerbSuche
    {
        $search = new FormationswettbewerbSuche();
        if ($request->request->getInt('turnierNr') > 0) {
            $search->setTurnierNr($request->request->getInt('turnierNr'));
        } else {
            $search->setVon($request->request->get('von'))
                ->setBis($request->request->get('bis'))
                ->setPlzVon($request->request->get('plzVon'))
                ->setPlzBis($request->request->get('plzBis'));

            if ($request->request->has('l')) {
                foreach($request->request->get('l') as $l) {
                    $search->addLandesverband($l);
                }
            }

            $search->setLiga($request->request->get('t1l'));
            $search->setTurnierart($request->request->get('t1a'));

            if ($request->request->has('seite')) {
                $search->setSeite($request->request->getInt('seite'));
            }
        }
        $search->setAnzahl(10);
        return $search;
    }

    private function createDefaultSearchParameters(): FormationswettbewerbSuche
    {
        $search = $this->service->createDefaultSuche();
        $search->setSeite(0);
        $search->setAnzahl(10);
        return $search;
    }
}
