<?php
declare(strict_types=1);

namespace Tanzsport\ContaoDtvApiBundle\Controller\Api;

use Contao\CoreBundle\Framework\ContaoFramework;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\Routing\Annotation\Route;
use Tanzsport\API\Turnier\EinzelwettbewerbDatenbankService;
use Tanzsport\API\Turnier\FormationswettbewerbDatenbankService;
use Tanzsport\API\Verein\VereinsDatenbankService;

/**
 * @Route("/_api/proxy/{service}", name=ProxyController::class)
 */
class ProxyController extends AbstractController
{
    private ContaoFramework $framework;
    private EinzelwettbewerbDatenbankService $einzelService;
    private FormationswettbewerbDatenbankService $formationsService;
    private VereinsDatenbankService $vereinsService;

    public function __construct(
        ContaoFramework $framework,
        EinzelwettbewerbDatenbankService $einzelService,
        FormationswettbewerbDatenbankService $formationsService,
        VereinsDatenbankService $vereinsService
    )
    {
        $this->framework = $framework;
        $this->framework->initialize(true);
        $this->einzelService = $einzelService;
        $this->formationsService = $formationsService;
        $this->vereinsService = $vereinsService;
    }

    public function __invoke(string $service, Request $request): Response
    {
        switch ($service) {
            case 'turnier-einzel':
                return $this->handleTurnierEinzel($request);
            case 'turnier-formation':
                return $this->handleTurnierFormation($request);
            case 'vereine':
                return $this->handleVereine($request);
            default:
                throw new BadRequestHttpException('Unknown service');
        }
    }

    private function handleTurnierEinzel(Request $request): Response
    {
        $query = $request->query->get('q');
        if (!$query) {
            throw new BadRequestHttpException('Query required');
        }

        switch ($query) {
            case 'sk':
                $param = $request->query->get('p');
                if (!$param) {
                    throw new BadRequestHttpException('Parameter required!');
                }
                return $this->json($this->einzelService->getStartklassen($param));
            case 'ta':
                $param1 = $request->query->get('p1');
                $param2 = $request->query->get('p2');
                if (!$param1 || !$param2) {
                    throw new BadRequestHttpException('Parameter 1 and 2 required!');
                }
                return $this->json($this->einzelService->getTurnierarten($param1, $param2))->setPrivate();
            default:
                throw new BadRequestHttpException('Unsupported query');
        }
    }

    private function handleTurnierFormation(Request $request): Response
    {
        $query = $request->query->get('q');
        if (!$query) {
            throw new BadRequestHttpException('Query required');
        }

        switch ($query) {
            case 'sl':
                $param = $request->query->get('p');
                if (!$param) {
                    throw new BadRequestHttpException('Parameter required!');
                }
                return $this->json($this->formationsService->getTurnierarten($param))->setPrivate();
            default:
                throw new BadRequestHttpException('Unsupported query');
        }
    }

    private function handleVereine(Request $request): Response
    {
        $query = $request->query->get('q');
        if (!$query) {
            throw new BadRequestHttpException('Query required');
        }

        switch ($query) {
            case 's':
                $param = $request->query->get('p');
                if (!$param) {
                    throw new BadRequestHttpException('Parameter required!');
                }
                return $this->json($this->vereinsService->findStandorte($param))->setPrivate();
            default:
                throw new BadRequestHttpException('Unsupported query');
        }
    }
}
