<?php
declare(strict_types=1);

namespace Tanzsport\ContaoDtvApiBundle\Controller\FrontendModule;

use Contao\CoreBundle\Controller\FrontendModule\AbstractFrontendModuleController;
use Contao\CoreBundle\ServiceAnnotation\FrontendModule;
use Contao\FrontendTemplate;
use Contao\ModuleModel;
use Contao\Template;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Tanzsport\API\Lehrgang\LehrgangsDatenbankService;
use Tanzsport\API\Lehrgang\LehrgangsListe;
use Tanzsport\API\Lehrgang\LehrgangSuche;
use Tanzsport\ContaoDtvApiBundle\UI\FormPagination;
use Tanzsport\Validation\BindingResult;

/**
 * @FrontendModule("td_lehrgangssuche",
 *     category="td",
 *     template="mod_lehrgangssuche_formular",
 *     renderer="forward"
 * )
 */
class LehrgangsSucheFrontendController extends AbstractFrontendModuleController
{

    private static string $formId = 'mod_lehrgangssuche_formular';

    private int $pageSize = 10;
    private LehrgangsDatenbankService $service;
    private LoggerInterface $contaoErrorLogger;

    public function __construct(LehrgangsDatenbankService $service, LoggerInterface $contaoErrorLogger)
    {
        $this->service = $service;
        $this->contaoErrorLogger = $contaoErrorLogger;
    }

    protected function getResponse(Template $template, ModuleModel $model, Request $request): ?Response
    {
        try {
            if ($request->isMethod('POST') && $request->request->get('FORM_SUBMIT') === self::$formId) {
                $search = $this->bindSearchParameters($request);
                $bindingResult = $this->service->validate($search);
                if (!$bindingResult->isError()) {
                    $result = $this->service->suchen($search);
                    return $this->createResponseResult($result);
                } else {
                    return $this->createResponseForm($template, $bindingResult, $search);
                }
            } else {
                return $this->createResponseForm($template);
            }
        } catch (\Throwable $e) {
            $this->contaoErrorLogger->log('error', 'Error querying Lehrgangsdatenbank', ['exception' => $e]);
            $template = new FrontendTemplate('mod_api_error');
            $template->message = $e->getMessage();
            return $template->getResponse()->setPrivate();
        }
    }

    private function createResponseForm(Template $template, ?BindingResult $bindingResult = null, ?LehrgangSuche $search = null): ?Response
    {
        if ($bindingResult === null) {
            $bindingResult = BindingResult::createEmpty();
        }
        if ($search === null) {
            $search = $this->createDefaultSearchParameters();
        }

        $parameters = $this->service->getParameters();

        $template->formId = self::$formId;
        $template->typen = $parameters['typen'];
        $template->ausrichter = $parameters['ausrichter'];
        $template->binding = $bindingResult;
        $template->suche = $search;

        return $template->getResponse()->setPrivate();
    }

    private function createResponseResult(LehrgangsListe $result): ?Response
    {
        $template = new FrontendTemplate('mod_lehrgangssuche_ergebnis');
        $template->ergebnis = $result;
        $template->pager = new FormPagination($result->getAnzahlGesamt(), 10, 7, 'seite', true);
        return $template->getResponse()->setPrivate();
    }

    private function bindSearchParameters(Request $request): LehrgangSuche
    {
        $search = new LehrgangSuche();

        $search->setVon($request->request->get('von'))->setBis($request->request->get('bis'));
        if ($request->request->has('typ')) {
            foreach ($request->request->get('typ') as $typ) {
                $search->addTyp($typ);
            }
        }
        if ($request->request->has('ausrichter')) {
            foreach ($request->request->get('ausrichter') as $ausrichter) {
                $search->addAusrichter($ausrichter);
            }
        }
        if ($request->request->has('seite')) {
            $search->setSeite($request->request->get('seite'));
        }

        $search->setAnzahl($this->pageSize);

        return $search;
    }

    private function createDefaultSearchParameters()
    {
        $suche = $this->service->createDefaultSearchParameters();
        $suche->setSeite(0);
        $suche->setAnzahl($this->pageSize);
        return $suche;
    }
}
