import { writable } from "svelte/store";
import type FilterInput from "../types/FilterInput";
import type RestResult from "../types/RestResult";
import type Tag from "../types/Tag";
import type Tags from "../types/Tags";
import { tags } from "./TagStore";

export function loadFilter(id: string) {
  filterInput.set(
    initFilter(
      JSON.parse(
        localStorage.getItem("searchfilter_" + id) || "{}"
      ) as Partial<FilterInput>
    )
  );

  filterInput.subscribe((val) =>
    localStorage.setItem("searchfilter_" + id, JSON.stringify(val))
  );

  tags.subscribe(validateFilters);
}

export function resetFilter() {
  filterInput.set(initFilter({}));
}

function validateTagExists(tag?: number, tags?: Tag[]) {
  return tags?.find((t) => t.id == tag) ? tag : undefined;
}

function validateFilters(tags: RestResult<Tags>) {
  if (tags.data?.tags) {
    filterInput.update((old) => ({
      ...old,
      leistungsklasse: validateTagExists(
        old.leistungsklasse,
        tags.data?.tags.LEISTUNGSKLASSEN
      ),
      turnierart: validateTagExists(
        old.turnierart,
        tags.data?.tags.TURNIERARTEN
      ),
      selectionTurnierform: old.selectionTurnierform.filter((i) =>
        tags.data?.tags.TURNIERFORMEN?.find((t) => t.id == i)
      ),
      selectionAltersklasse: old.selectionAltersklasse.filter((i) =>
        tags.data?.tags.ALTERSKLASSEN?.find((t) => t.id == i)
      ),
      selectionLtv: old.selectionLtv.filter((i) =>
        tags.data?.tags.LANDESVERBAND?.find((t) => t.id == i)
      ),
      selectionWettbewerbsart: old.selectionWettbewerbsart.filter((i) =>
        tags.data?.tags.WETTBEWERBSART?.find((t) => t.id == i)
      ),
      selectionTurnierart: old.selectionTurnierart.filter((i) =>
        tags.data?.tags.TURNIERARTEN?.find((t) => t.id == i)
      ),
    }));
  }
}

function initFilter(initialData: Partial<FilterInput>): FilterInput {
  return {
    ...initialData,
    conjunction: !!initialData.conjunction,
    selectionTurnierform: initialData.selectionTurnierform || [],
    selectionLtv: initialData.selectionLtv || [],
    selectionAltersklasse: initialData.selectionAltersklasse || [],
    selectionTurnierart: initialData.selectionTurnierart || [],
    selectionWettbewerbsart: initialData.selectionWettbewerbsart || [],
    dateFrom: initialData.dateFrom || new Date().toISOString().substring(0, 10),
    dateTo:
      initialData.dateTo ||
      new Date(Date.now() + 180 * 24 * 3600 * 1000)
        .toISOString()
        .substring(0, 10),
  };
}

export const filterInput = writable<FilterInput>(initFilter({}));
