<?php
declare(strict_types=1);

namespace Tanzsport\ContaoDtvApiBundle\Controller\Pdf;

use Psr\Log\LoggerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\HttpKernel\Exception\HttpException;
use Symfony\Component\Routing\Annotation\Route;
use Tanzsport\API\Turnier\EinzelwettbewerbDatenbankService;
use Tanzsport\API\Turnier\FormationswettbewerbDatenbankService;
use Tanzsport\API\Turnier\JmdWettbewerbDatenbankService;

/**
 * @Route("/_turniersuche-pdf-export/{service}",
 *     name=TurnierSuchePdfExportController::class,
 *     methods={"POST"}
 * )
 */
class TurnierSuchePdfExportController extends AbstractController
{

    private EinzelwettbewerbDatenbankService $einzelService;
    private FormationswettbewerbDatenbankService $formationsService;
    private JmdWettbewerbDatenbankService $jmdService;
    private LoggerInterface $contaoErrorLogger;


    public function __construct(
        EinzelwettbewerbDatenbankService     $einzelService,
        FormationswettbewerbDatenbankService $formationsService,
        JmdWettbewerbDatenbankService        $jmdService,
        LoggerInterface                      $contaoErrorLogger
    )
    {
        $this->einzelService = $einzelService;
        $this->formationsService = $formationsService;
        $this->jmdService = $jmdService;
        $this->contaoErrorLogger = $contaoErrorLogger;
    }

    public function __invoke(string $service, Request $request): Response
    {
        try {
            if ($request->request->has('search')) {
                switch ($service) {
                    case 'einzel':
                        $search = $this->einzelService->unserialize($request->request->get('search'));
                        $bindingResult = $this->einzelService->validate($search);
                        break;
                    case 'formation':
                        $search = $this->formationsService->unserialize($request->request->get('search'));
                        $bindingResult = $this->formationsService->validate($search);
                        break;
                    case 'jmc':
                        $search = $this->jmdService->unserialize($request->request->get('search'));
                        $bindingResult = $this->jmdService->validate($search);
                        break;
                    default:
                        throw new BadRequestHttpException("Service {$service} is not supported");
                }

                if ($bindingResult->isError()) {
                    throw new BadRequestHttpException('Search is invalid');
                }

                switch ($service) {
                    case 'einzel':
                        $binary = $this->einzelService->exportieren($search);
                        break;
                    case 'formation':
                        $binary = $this->formationsService->exportieren($search);
                        break;
                    case 'jmc':
                        $binary = $this->jmdService->exportieren($search);
                        break;
                    default:
                        throw new BadRequestHttpException("Service {$service} is not supported");
                }

                return $binary->handle(function ($response) {
                    $r = new Response();
                    $r->headers->set('Content-Type', 'application/pdf');
                    $r->headers->set('Content-Disposition', $r->headers->makeDisposition(ResponseHeaderBag::DISPOSITION_INLINE, 'turniere.pdf'));
                    $r->sendHeaders();
                    $r->setContent((string)$response->getBody());
                    $r->setPrivate();
                    return $r;
                });
            } else {
                throw new BadRequestHttpException('Search is required');
            }
        } catch (\Throwable $e) {
            if ($e instanceof BadRequestHttpException) {
                throw $e;
            }
            $this->contaoErrorLogger->log('error', 'Error exporting PDF', ['exception' => $e]);
            throw new HttpException(500, $e->getMessage());
        }
    }
}
