<?php
declare(strict_types=1);

namespace Tanzsport\API\Turnier;

use Tanzsport\API\BinaryResponse;
use Tanzsport\API\Turnier\Resource\EinzelwettbewerbResource;
use Tanzsport\API\Turnier\Validation\EinzelwettbewerbSucheValidator;
use Tanzsport\Validation\BindingResult;

class EinzelwettbewerbDatenbankService extends AbstractDatenbankService implements TurnierSucheService
{

	private EinzelwettbewerbResource $einzelwettbewerbResource;
	private EinzelwettbewerbSucheValidator $validator;
	private ?\stdClass $parameterCache = null;
	/**
	 * @var Array<string>
	 */
	private $startklassenCache = [];
	/**
	 * @var Array<string>
	 */
	private $turnierartenCache = [];

	public function __construct($endpoint, $enableCache = false, $cacheDir = null, $cookieName = null, $cookieExpirationDays = null, $encryptionKey = null)
	{
		parent::__construct($endpoint, $enableCache, $cacheDir, $cookieName, $cookieExpirationDays, $encryptionKey);

		$this->einzelwettbewerbResource = new EinzelwettbewerbResource($endpoint, $enableCache, $cacheDir);
		$this->validator = new EinzelwettbewerbSucheValidator();
	}

	private function getAlleParameter()
	{
		if ($this->parameterCache == null) {
			$this->parameterCache = (object)$this->einzelwettbewerbResource->getAlleParameter();
		}
		return $this->parameterCache;
	}

	public function getLandesverbaende()
	{
		return $this->sortLandeverbande($this->getAlleParameter()->landesverbaende);
	}

	public function getStartgruppen()
	{
		return $this->getAlleParameter()->startgruppen;
	}

	public function getStartklassen($startgruppe)
	{
		// @codeCoverageIgnoreStart
		if ($startgruppe == null) {
			throw new \InvalidArgumentException("Startgruppe erforderlich!");
		}
		// @codeCoverageIgnoreEnd

		$startgruppe = strtoupper(trim($startgruppe));

		if (isset($this->startklassenCache[$startgruppe])) {
			return $this->startklassenCache[$startgruppe];
		} else {
			$regeln = $this->getRegeln();
			$startklassen = array();
			foreach ($regeln as $regel) {
				$r = (object)$regel;
				if (strtoupper($r->startgruppe) == $startgruppe) {
					if (!in_array($r->startklasse, $startklassen)) {
						$startklassen[] = $r->startklasse;
					}
				}
			}
			$this->startklassenCache[$startgruppe] = $startklassen;
			return $startklassen;
		}
	}

	public function getTurnierarten($startgruppe, $startklasse)
	{
		// @codeCoverageIgnoreStart
		if ($startgruppe == null) {
			throw new \InvalidArgumentException("Startgruppe erforderlich!");
		}
		if ($startklasse == null) {
			throw new \InvalidArgumentException("Startklasse erforderlich!");
		}
		// @codeCoverageIgnoreEnd

		$startgruppe = strtoupper(trim($startgruppe));
		$startklasse = strtoupper(trim($startklasse));

		if (isset($this->turnierartenCache[$startgruppe][$startklasse])) {
			return $this->turnierartenCache[$startgruppe][$startklasse];
		} else {
			$regeln = $this->getRegeln();
			$turnierarten = array();
			foreach ($regeln as $regel) {
				$r = (object)$regel;
				if (strtoupper($r->startgruppe) == $startgruppe && strtoupper($r->startklasse) == $startklasse) {
					if (!in_array($r->turnierart, $turnierarten)) {
						$turnierarten[] = $r->turnierart;
					}
				}
			}
			$this->turnierartenCache[$startgruppe][$startklasse] = $turnierarten;
			return $turnierarten;
		}
	}

	public function getTypen()
	{
		return $this->getAlleParameter()->typen;
	}

	public function createDefaultSuche(): EinzelwettbewerbSuche
	{
		$suche = null;

		if ($this->clientPersistenceHandler != null) {
			$suche = $this->clientPersistenceHandler->retrieve();
		}

		if ($suche == null) {
			$dt = new \DateTime();

			$suche = new EinzelwettbewerbSuche();
			$suche->setVon($dt->format('Y-m-d'))->setBis($dt->add(new \DateInterval('P2M'))->format('Y-m-d'));
		}

		return $suche;
	}

	public function suchen(EinzelwettbewerbSuche $suche): Suchergebnis
	{
		$this->validateAndFail($suche);

		if ($suche->getTurnierNr() > 0) {
			$veranstaltung = $this->einzelwettbewerbResource->findVeranstaltungMitTurnierNr($suche->getTurnierNr());
			if ($veranstaltung) {
				return new Suchergebnis($this->toPageArray([$veranstaltung]));
			} else {
				return new Suchergebnis($this->toPageArray([]));
			}
		} else {
			if ($this->clientPersistenceHandler != null) {
				$this->clientPersistenceHandler->store($suche);
			}

			$apiResult = $this->einzelwettbewerbResource->findJson(
				$this->convertDate($suche->getVon()), $this->convertDate($suche->getBis()),
				$suche->getPlzVon(), $suche->getPlzBis(),
				$suche->getTypen(), $suche->getLandesverbaende(),
				$suche->getTurnier1()->gruppe, $suche->getTurnier1()->klasse, $suche->getTurnier1()->turnierart,
				$suche->getTurnier2()->gruppe, $suche->getTurnier2()->klasse, $suche->getTurnier2()->turnierart,
				$suche->getKonjunktion() ? true : false,
				intval($suche->getSeite()),
				intval($suche->getAnzahl())
			);

			return new Suchergebnis($apiResult);
		}
	}

	public function exportieren(EinzelwettbewerbSuche $suche): ?BinaryResponse
	{
		$this->validateAndFail($suche);

		if ($suche->getTurnierNr() > 0) {
			$response = $this->einzelwettbewerbResource->findVeranstaltungMitTurnierNrPdf($suche->getTurnierNr());
			if ($response) {
				return new BinaryResponse($response);
			} else {
				return null;
			}
		} else {
			$response = $this->einzelwettbewerbResource->findPdf(
				$this->convertDate($suche->getVon()), $this->convertDate($suche->getBis()),
				$suche->getPlzVon(), $suche->getPlzBis(),
				$suche->getTypen(), $suche->getLandesverbaende(),
				$suche->getTurnier1()->gruppe, $suche->getTurnier1()->klasse, $suche->getTurnier1()->turnierart,
				$suche->getTurnier2()->gruppe, $suche->getTurnier2()->klasse, $suche->getTurnier2()->turnierart,
				$suche->getKonjunktion() ? true : false
			);

			return new BinaryResponse($response);
		}
	}

	public function validate(EinzelwettbewerbSuche $suche): BindingResult
	{
		return $this->validator->validate($suche);
	}

	private function getRegeln()
	{
		return $this->getAlleParameter()->regeln;
	}

	private function validateAndFail(EinzelwettbewerbSuche $suche)
	{
		$bindingResult = $this->validate($suche);
		if ($bindingResult->isError()) {
			throw new \InvalidArgumentException("Suchparameter sind ungültig!");
		}
	}
}
