<?php
namespace Tanzsport\API\Lehrgang;

use Tanzsport\API\Lehrgang\Resource\LehrgaengeResource;
use Tanzsport\API\Lehrgang\Resource\ParameterResource;
use Tanzsport\API\Lehrgang\Validation\LehrgangSucheValidator;
use Tanzsport\Util\ClientPersistenceHandler;
use Tanzsport\Util\Serializer;
use Tanzsport\Validation\BindingResult;

class LehrgangsDatenbankService
{

	const REGEX_DATUM1 = '/(\d{1,2})\.(\d{1,2})\.(\d{4})/'; //d.m.Y
	const REGEX_DATUM2 = '/(\d{4})\-(\d{1,2})\-(\d{1,2})/'; //Y-m-d
	const REGEX_DATUM3 = '/(\d{4})\/(\d{1,2})\/(\d{1,2})/'; //Y/m/d

	/**
	 * @var string
	 */
	private $endpoint;

	/**
	 * @var ParameterResource
	 */
	private $parameterResource;

	/**
	 * @var LehrgaengeResource
	 */
	private $lehrgaengeResource;

	/**
	 * @var LehrgangsSucheValidator
	 */
	private $validator;

	/**
	 * @var Serializer
	 */
	protected $serializer;

	/**
	 * @var ClientPersistenceHandler
	 */
	private $clientPersistenceHandler;

	public function __construct($endpoint, $enableCache = false, $cacheDir = null, $cookieName = null, $cookieExpirationDays = null, $encryptionKey = null)
	{
		// @codeCoverageIgnoreStart
		if ($endpoint == null) {
			throw new \InvalidArgumentException("API-Endpunkt erforderlich!");
		}
		// @codeCoverageIgnoreEnd

		$this->endpoint = $endpoint;

		$this->parameterResource = new ParameterResource($endpoint, $enableCache, $cacheDir);
		$this->lehrgaengeResource = new LehrgaengeResource($endpoint, $enableCache, $cacheDir);
		$this->validator = new LehrgangSucheValidator();
		$this->serializer = new Serializer($encryptionKey);

		if ($cookieName != null && $cookieExpirationDays > 0) {
			$this->clientPersistenceHandler = new ClientPersistenceHandler($cookieName, $this->serializer, $cookieExpirationDays);
		}
	}

	public function getParameters()
	{
		$parameter = $this->parameterResource->getAll();
		uasort($parameter['ausrichter'], function($a, $b) {
			return strcmp($a, $b);
		});
		return $parameter;
	}

	/**
	 * @param LehrgangSuche $suche
	 * @return LehrgangsListe
	 */
	public function suchen(LehrgangSuche $suche)
	{
		$this->validateAndFail($suche);

		if ($this->clientPersistenceHandler != null) {
			$this->clientPersistenceHandler->store($suche);
		}

		return new LehrgangsListe($this->lehrgaengeResource->find(
			$this->convertDate($suche->getVon()),
			$this->convertDate($suche->getBis()),
			$suche->getTypen(),
			$suche->getAusrichter(),
			intval($suche->getSeite()),
			intval($suche->getAnzahl())
		));
	}

	/**
	 * @param LehrgangSuche $suche
	 * @return BindingResult
	 */
	public function validate(LehrgangSuche $suche)
	{
		return $this->validator->validate($suche);
	}

	/**
	 * @param $value
	 * @return string
	 * @throws \InvalidArgumentException
	 */
	public function serialize($value)
	{
		// @codeCoverageIgnoreStart
		if ($value == null) {
			throw new \InvalidArgumentException("Wert erforderlich!");
		}
		// @codeCoverageIgnoreEnd

		return $this->serializer->serializeAndEncrypt($value);
	}

	/**
	 * @param $serializedValue
	 * @return mixed
	 */
	public function unserialize($serializedValue)
	{
		return $this->serializer->decryptAndDeserialize($serializedValue);
	}

	public function createDefaultSearchParameters()
	{
		$suche = null;

		if ($this->clientPersistenceHandler != null) {
			$suche = $this->clientPersistenceHandler->retrieve();
		}

		if ($suche == null) {
			$dt = new \DateTime();

			$suche = new LehrgangSuche();
			$suche->setVon($dt->format('Y-m-d'))->setBis($dt->add(new \DateInterval('P2M'))->format('Y-m-d'));
		}

		return $suche;
	}

	private function validateAndFail(LehrgangSuche $suche)
	{
		$br = $this->validate($suche);
		if ($br->isError()) {
			throw new \InvalidArgumentException("Suchparameter sind ungültig!");
		}
	}

	protected function convertDate($input)
	{
		$matches = [];
		if (preg_match(self::REGEX_DATUM1, $input, $matches)) {
			return $input;
		} else if (preg_match(self::REGEX_DATUM2, $input, $matches)) {
			return $matches[3] . "." . $matches[2] . "." . $matches[1];
		} else if (preg_match(self::REGEX_DATUM3, $input, $matches)) {
			return $matches[3] . "." . $matches[2] . "." . $matches[1];
		} else {
			throw new \InvalidArgumentException("$input ist kein gültiges Datum!");
		}
	}
}

