<?php
namespace Tanzsport\Util;

class ClientPersistenceHandler
{

	/**
	 * @var string
	 */
	private $symbolicName;

	/**
	 * @var bool
	 */
	private $useCookies = true;

	/**
	 * @var int
	 */
	private $expirationDays;

	/**
	 * @var array
	 */
	private $memory = array();

	/**
	 * @var Serializer
	 */
	private $serializer;

	public function __construct($symbolicName, Serializer $serializer, $expirationDays = 365)
	{
		// @codeCoverageIgnoreStart
		if ($symbolicName == null) {
			throw new \InvalidArgumentException("Symbolischer Name erforderlich!");
		}
		if ($expirationDays == null) {
			throw new \InvalidArgumentException("Speicherdauer erforderlich!");
		}
		// @codeCoverageIgnoreEnd

		$this->symbolicName = $symbolicName;
		$this->serializer = $serializer;
		$this->expirationDays = $expirationDays;

		if (php_sapi_name() == 'cli') {
			$this->useCookies = false;
		}
	}

	public function store($value)
	{
		// @codeCoverageIgnoreStart
		if ($value == null) {
			throw new \InvalidArgumentException("Wert erforderlich!");
		}
		// @codeCoverageIgnoreEnd

		// @codeCoverageIgnoreStart
		if ($this->useCookies) {
			$this->storeAsCookie($value);
		} else {
			// @codeCoverageIgnoreEnd
			$this->storeInMemory($value);
		}
	}

	public function remove()
	{
		// @codeCoverageIgnoreStart
		if ($this->useCookies) {
			setcookie($this->symbolicName, '', time() - 3600);
		} else {
			// @codeCoverageIgnoreEnd
			unset($this->memory[$this->symbolicName]);
		}
	}

	public function retrieve()
	{
		// @codeCoverageIgnoreStart
		if ($this->useCookies) {
			if (isset($_COOKIE[$this->symbolicName])) {
				return $this->serializer->decryptAndDeserialize($_COOKIE[$this->symbolicName]);
			}
		} else {
			// @codeCoverageIgnoreEnd
			if (isset($this->memory[$this->symbolicName])) {
				return $this->serializer->decryptAndDeserialize($this->memory[$this->symbolicName]);
			}
		}

		return null;
	}

	/**
	 * @codeCoverageIgnore
	 */
	private function storeAsCookie($value)
	{
		setcookie($this->symbolicName, $this->serializer->serializeAndEncrypt($value), time() + ($this->expirationDays * 24 * 60 * 60));
	}

	private function storeInMemory($value)
	{
		$this->memory[$this->symbolicName] = $this->serializer->serializeAndEncrypt($value);
	}

}