<?php
declare(strict_types=1);

namespace Tanzsport\API\Turnier;

use Tanzsport\API\BinaryResponse;
use Tanzsport\API\Turnier\Resource\JmdWettbewerbResource;
use Tanzsport\API\Turnier\Validation\JmdWettbewerbSucheValidator;
use Tanzsport\Validation\BindingResult;

class JmdWettbewerbDatenbankService extends AbstractDatenbankService implements TurnierSucheService
{

	private JmdWettbewerbResource $jmdWettbewerbResource;
	private JmdWettbewerbSucheValidator $validator;
	private ?\stdClass $parameterCache = null;

	public function __construct($endpoint, $enableCache = false, $cacheDir = null, $cookieName = null, $cookieExpirationDays = null, $encryptionKey = null)
	{
		parent::__construct($endpoint, $enableCache, $cacheDir, $cookieName, $cookieExpirationDays, $encryptionKey);

		$this->jmdWettbewerbResource = new JmdWettbewerbResource($endpoint, $enableCache, $cacheDir);
		$this->validator = new JmdWettbewerbSucheValidator();
	}

	private function getAlleParameter()
	{
		if ($this->parameterCache == null) {
			$this->parameterCache = (object)$this->jmdWettbewerbResource->getAlleParameter();
		}
		return $this->parameterCache;
	}

	public function getLandesverbaende()
	{
		return $this->sortLandeverbande($this->getAlleParameter()->landesverbaende);
	}

	public function getTypen()
	{
		return $this->getAlleParameter()->typen;
	}

	public function getWettbewerbsarten()
	{
		return $this->getAlleParameter()->wettbewerbsarten;
	}

	public function createDefaultSuche(): JmdWettbewerbSuche
	{
		$suche = null;

		if ($this->clientPersistenceHandler != null) {
			$suche = $this->clientPersistenceHandler->retrieve();
		}

		if ($suche == null) {
			$dt = new \DateTime();

			$suche = new JmdWettbewerbSuche();
			$suche->setVon($dt->format('Y-m-d'))->setBis($dt->add(new \DateInterval('P2M'))->format('Y-m-d'));
		}

		return $suche;
	}

	public function suchen(JmdWettbewerbSuche $suche): Suchergebnis
	{
		$this->validateAndFail($suche);

		if ($suche->getTurnierNr() > 0) {
			$veranstaltung = $this->jmdWettbewerbResource->findVeranstaltungMitTurnierNr($suche->getTurnierNr());
			if ($veranstaltung) {
				return new Suchergebnis($this->toPageArray([$veranstaltung]));
			} else {
				return new Suchergebnis($this->toPageArray([]));
			}
		} else {
			if ($this->clientPersistenceHandler != null) {
				$this->clientPersistenceHandler->store($suche);
			}

			$apiResult = $this->jmdWettbewerbResource->findJson(
				$this->convertDate($suche->getVon()), $this->convertDate($suche->getBis()),
				$suche->getPlzVon(), $suche->getPlzBis(),
				$suche->getTypen(),
				$suche->getLandesverbaende(),
				$suche->getWettbewerbsarten(),
				intval($suche->getSeite()),
				intval($suche->getAnzahl())
			);

			return new Suchergebnis($apiResult);
		}
	}

	public function exportieren(JmdWettbewerbSuche $suche): ?BinaryResponse
	{
		$this->validateAndFail($suche);

		if ($suche->getTurnierNr() > 0) {
			$response = $this->jmdWettbewerbResource->findVeranstaltungMitTurnierNrPdf($suche->getTurnierNr());
			if ($response) {
				return new BinaryResponse($response);
			} else {
				return null;
			}
		} else {
			$response = $this->jmdWettbewerbResource->findPdf(
				$this->convertDate($suche->getVon()), $this->convertDate($suche->getBis()),
				$suche->getPlzVon(), $suche->getPlzBis(),
				$suche->getTypen(),
				$suche->getLandesverbaende(),
				$suche->getWettbewerbsarten()
			);

			return new BinaryResponse($response);
		}
	}

	public function validate(JmdWettbewerbSuche $suche): BindingResult
	{
		return $this->validator->validate($suche);
	}

	private function validateAndFail(JmdWettbewerbSuche $suche)
	{
		$bindingResult = $this->validate($suche);
		if ($bindingResult->isError()) {
			throw new \InvalidArgumentException("Suchparameter sind ungültig!");
		}
	}
}
