<?php
namespace Tanzsport\API\Turnier;

use Tanzsport\Util\ClientPersistenceHandler;
use Tanzsport\Util\Serializer;

class AbstractDatenbankService
{

	const REGEX_DATUM1 = '/(\d{1,2})\.(\d{1,2})\.(\d{4})/'; //d.m.Y
	const REGEX_DATUM2 = '/(\d{4})\-(\d{1,2})\-(\d{1,2})/'; //Y-m-d
	const REGEX_DATUM3 = '/(\d{4})\/(\d{1,2})\/(\d{1,2})/'; //Y/m/d

	/**
	 * @var string
	 */
	protected $endpoint;

	/**
	 * @var Serializer
	 */
	protected $serializer;

	/**
	 * @var ClientPersistenceHandler
	 */
	protected $clientPersistenceHandler;

	public function __construct($endpoint, $enableCache = false, $cacheDir = null, $cookieName = null, $cookieExpirationDays = null, $encryptionKey = null)
	{
		// @codeCoverageIgnoreStart
		if ($endpoint == null) {
			throw new \InvalidArgumentException("API-Endpunkt erforderlich!");
		}
		// @codeCoverageIgnoreEnd

		$this->endpoint = $endpoint;
		$this->serializer = new Serializer($encryptionKey);

		if ($cookieName != null && $cookieExpirationDays > 0) {
			$this->clientPersistenceHandler = new ClientPersistenceHandler($cookieName, $this->serializer, $cookieExpirationDays);
		}
	}

	public function reset()
	{
		if ($this->clientPersistenceHandler != null) {
			$this->clientPersistenceHandler->remove();
		}
	}

	/**
	 * @param $value
	 * @return string
	 * @throws \InvalidArgumentException
	 */
	public function serialize($value)
	{
		// @codeCoverageIgnoreStart
		if ($value == null) {
			throw new \InvalidArgumentException("Wert erforderlich!");
		}
		// @codeCoverageIgnoreEnd

		return $this->serializer->serializeAndEncrypt($value);
	}

	/**
	 * @param $serializedValue
	 * @return mixed
	 */
	public function unserialize($serializedValue)
	{
		return $this->serializer->decryptAndDeserialize($serializedValue);
	}

	/**
	 * @param array $array
	 * @return array
	 */
	protected function toPageArray(array $array = null)
	{
		return [
			'daten' => $array,
			'seitenLaenge' => count($array),
			'seite' => 1,
			'anzahlGesamt' => count($array),
			'anzahlDatensaetze' => count($array),
			'seitenAzahl' => 1
		];
	}

	protected function convertDate($input)
	{
		$matches = [];
		if (preg_match(self::REGEX_DATUM1, $input, $matches)) {
			return $input;
		} else if (preg_match(self::REGEX_DATUM2, $input, $matches)) {
			return $matches[3] . "." . $matches[2] . "." . $matches[1];
		} else if (preg_match(self::REGEX_DATUM3, $input, $matches)) {
			return $matches[3] . "." . $matches[2] . "." . $matches[1];
		} else {
			throw new \InvalidArgumentException("$input ist kein gültiges Datum!");
		}
	}

	protected function sortLandeverbande(array $landesverbaende)
	{
		uasort($landesverbaende, function($a, $b) {
			return strcmp($a, $b);
		});
		return $landesverbaende;
	}
}