<?php
declare(strict_types=1);

namespace Tanzsport\API;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use Psr\Http\Message\ResponseInterface;

class AbstractResource
{

	private string $endpoint;
	private Client $client;
	private ?string $cacheDir = null;

	public function __construct(string $endpoint, bool $enableCache = false, ?string $cacheDir = null)
	{
		// @codeCoverageIgnoreStart
		if ($endpoint == null) {
			throw new \InvalidArgumentException("Endpoint erforderlich!");
		}
		// @codeCoverageIgnoreEnd

		$this->endpoint = rtrim($endpoint, " \n\r\t\v\x00/") . '/';
		$this->client = new Client([
			'base_uri' => $this->endpoint,
			'decode_content' => true,
			'verify' => false
		]);

		// @codeCoverageIgnoreStart
		/*
		if ($enableCache) {
			if ($cacheDir == null) {
				$this->cacheDir = sys_get_temp_dir() . '/tdapi';
			} else {
				$this->cacheDir = $cacheDir . '/tdapi';
			}
			$this->autocreateCacheDir();

			$this->cachePlugin = new CachePlugin(array(
				'storage' => new DefaultCacheStorage(new DoctrineCacheAdapter(new FilesystemCache($this->cacheDir)))
			));

			$this->client->addSubscriber($this->cachePlugin);
		}
		*/
		// @codeCoverageIgnoreEnd
	}

	/**
	 * @param string $relativeUrl
	 * @param array $options
	 * @return ResponseInterface
	 * @throws GuzzleException
	 */
	protected function get(string $relativeUrl, array $options = []): ResponseInterface
	{
		$queryString = '';
		if (isset($options['query']) && count($options['query']) > 0) {
			$queryString = '?';
			$first = true;
			foreach ($options['query'] as $key => $value) {
				$values = $value;
				if (!is_array($value)) {
					$values = [$value];
				}
				foreach ($values as $v) {
					if (!$first) {
						$queryString .= '&';
					}
					$queryString .= urlencode($key) . '=' . urlencode(strval($v));
					$first = false;
				}
			}
			unset($options['query']);
		}
		$relativeUrl = $relativeUrl . $queryString;
		return $this->client->get($relativeUrl, $options);
	}

	protected function getJson($relativeUrl, $options = [])
	{
		return $this->parseJson($this->get($relativeUrl, $options));
	}

	protected function parseJson(ResponseInterface $response)
	{
		if ($response->getStatusCode() === 200) {
			try {
				return json_decode((string)$response->getBody(), true, 512, JSON_THROW_ON_ERROR);
			} catch (\Exception $e) {
				return null;
			}
		} else {
			return null;
		}
	}

	/**
	 * @codeCoverageIgnore
	 */
	private function autocreateCacheDir()
	{
		if (!is_dir($this->cacheDir)) {
			mkdir($this->cacheDir, 0777, true);
		}
	}
}
