<?php
namespace Tanzsport\API\Turnier;

use Tanzsport\API\BinaryResponse;
use Tanzsport\API\Turnier\Resource\JmdWettbewerbResource;
use Tanzsport\API\Turnier\Validation\JmdWettbewerbSucheValidator;

class JmdWettbewerbDatenbankService extends AbstractDatenbankService
{

	/**
	 * @var JmdWettbewerbResource
	 */
	private $jmdWettbewerbResource;

	/**
	 * @var JmdWettbewerbSucheValidator
	 */
	private $validator;

	/**
	 * @var \stdClass
	 */
	private $parameterCache;

	public function __construct($endpoint, $enableCache = false, $cacheDir = null, $cookieName = null, $cookieExpirationDays = null, $encryptionKey = null)
	{
		parent::__construct($endpoint, $enableCache, $cacheDir, $cookieName, $cookieExpirationDays, $encryptionKey);

		$this->jmdWettbewerbResource = new JmdWettbewerbResource($endpoint, $enableCache, $cacheDir);
		$this->validator = new JmdWettbewerbSucheValidator();
	}

	private function getAlleParameter()
	{
		if ($this->parameterCache == null) {
			$this->parameterCache = (object)$this->jmdWettbewerbResource->getAlleParameter();
		}
		return $this->parameterCache;
	}

	public function getLandesverbaende()
	{
		return $this->sortLandeverbande($this->getAlleParameter()->landesverbaende);
	}

	public function getTypen()
	{
		return $this->getAlleParameter()->typen;
	}

	public function getWettbewerbsarten()
	{
		return $this->getAlleParameter()->wettbewerbsarten;
	}

	/**
	 * @return JmdWettbewerbSuche
	 */
	public function createDefaultSuche()
	{
		$suche = null;

		if ($this->clientPersistenceHandler != null) {
			$suche = $this->clientPersistenceHandler->retrieve();
		}

		if ($suche == null) {
			$dt = new \DateTime();

			$suche = new JmdWettbewerbSuche();
			$suche->setVon($dt->format('Y-m-d'))->setBis($dt->add(new \DateInterval('P2M'))->format('Y-m-d'));
		}

		return $suche;
	}

	/**
	 * @param JmdWettbewerbSuche $suche
	 * @return Suchergebnis
	 */
	public function suchen(JmdWettbewerbSuche $suche)
	{
		$this->validateAndFail($suche);

		if ($suche->getTurnierNr() > 0) {
			$veranstaltung = $this->jmdWettbewerbResource->findVeranstaltungMitTurnierNr($suche->getTurnierNr());
			if ($veranstaltung) {
				return new Suchergebnis($this->toPageArray([$veranstaltung->json()]));
			} else {
				return new Suchergebnis($this->toPageArray([]));
			}
		} else {
			if ($this->clientPersistenceHandler != null) {
				$this->clientPersistenceHandler->store($suche);
			}

			$apiResult = $this->jmdWettbewerbResource->findJson(
				$this->convertDate($suche->getVon()), $this->convertDate($suche->getBis()),
				$suche->getPlzVon(), $suche->getPlzBis(),
				$suche->getTypen(),
				$suche->getLandesverbaende(),
				$suche->getWettbewerbsarten(),
				intval($suche->getSeite()),
				intval($suche->getAnzahl())
			);

			return new Suchergebnis($apiResult);
		}
	}

	/**
	 * @param JmdWettbewerbSuche $suche
	 * @return BinaryResponse
	 */
	public function exportieren(JmdWettbewerbSuche $suche)
	{
		$this->validateAndFail($suche);

		if ($suche->getTurnierNr() > 0) {
			$response = $this->jmdWettbewerbResource->findVeranstaltungMitTurnierNrPdf($suche->getTurnierNr());
			if ($response) {
				return new BinaryResponse($response);
			} else {
				return null;
			}
		} else {
			$response = $this->jmdWettbewerbResource->findPdf(
				$this->convertDate($suche->getVon()), $this->convertDate($suche->getBis()),
				$suche->getPlzVon(), $suche->getPlzBis(),
				$suche->getTypen(),
				$suche->getLandesverbaende(),
				$suche->getWettbewerbsarten()
			);

			return new BinaryResponse($response);
		}
	}

	/**
	 * @param JmdWettbewerbSuche $suche
	 * @return \Tanzsport\Validation\BindingResult
	 */
	public function validate(JmdWettbewerbSuche $suche)
	{
		return $this->validator->validate($suche);
	}

	private function validateAndFail(JmdWettbewerbSuche $suche)
	{
		$br = $this->validate($suche);
		if ($br->isError()) {
			throw new \InvalidArgumentException("Suchparameter sind ungültig!");
		}
	}
}
