<?php
namespace Tanzsport\API\Turnier;

use Tanzsport\API\BinaryResponse;
use Tanzsport\API\Turnier\Resource\FormationswettbewerbResource;
use Tanzsport\API\Turnier\Validation\FormationswettbewerbSucheValidator;

class FormationswettbewerbDatenbankService extends AbstractDatenbankService
{

	/**
	 * @var FormationswettbewerbResource
	 */
	private $formationswettbewerbResource;

	/**
	 * @var Validation\FormationswettbewerbSucheValidator
	 */
	private $validator;

	/**
	 * @var \stdClass
	 */
	private $parameterCache;

	/**
	 * @var array
	 */
	private $turnierartenCache = array();

	public function __construct($endpoint, $enableCache = false, $cacheDir = null, $cookieName = null, $cookieExpirationDays = null, $encryptionKey = null)
	{
		parent::__construct($endpoint, $enableCache, $cacheDir, $cookieName, $cookieExpirationDays, $encryptionKey);

		$this->formationswettbewerbResource = new FormationswettbewerbResource($endpoint, $enableCache, $cacheDir);
		$this->validator = new FormationswettbewerbSucheValidator();
	}

	private function getAlleParameter()
	{
		if ($this->parameterCache == null) {
			$this->parameterCache = (object)$this->formationswettbewerbResource->getAlleParameter();
		}
		return $this->parameterCache;
	}

	public function getLandesverbaende()
	{
		return $this->sortLandeverbande($this->getAlleParameter()->landesverbaende);
	}

	public function getLigen()
	{
		return $this->getAlleParameter()->ligen;
	}

	public function getTurnierarten($liga)
	{
		// @codeCoverageIgnoreStart
		if ($liga == null) {
			throw new \InvalidArgumentException("Liga erforderlich!");
		}
		// @codeCoverageIgnoreEnd

		$liga = strtoupper(trim($liga));

		if (isset($this->turnierartenCache[$liga])) {
			return $this->turnierartenCache[$liga];
		} else {
			$regeln = $this->getRegeln();
			$turnierarten = array();
			foreach ($regeln as $regel) {
				$r = (object)$regel;
				if (strtoupper($r->liga) == $liga) {
					if (!in_array($r->turnierart, $turnierarten)) {
						$turnierarten[] = $r->turnierart;
					}
				}
			}
			$this->turnierartenCache[$liga] = $turnierarten;
			return $turnierarten;
		}
	}

	/**
	 * @return FormationswettbewerbSuche
	 */
	public function createDefaultSuche()
	{
		$suche = null;

		if ($this->clientPersistenceHandler != null) {
			$suche = $this->clientPersistenceHandler->retrieve();
		}

		if ($suche == null) {
			$dt = new \DateTime();

			$suche = new FormationswettbewerbSuche();
			$suche->setVon($dt->format('Y-m-d'))->setBis($dt->add(new \DateInterval('P2M'))->format('Y-m-d'));
		}

		return $suche;
	}

	/**
	 * @param FormationswettbewerbSuche $suche
	 * @return Suchergebnis
	 */
	public function suchen(FormationswettbewerbSuche $suche)
	{
		$this->validateAndFail($suche);

		if ($suche->getTurnierNr() > 0) {
			$veranstaltung = $this->formationswettbewerbResource->findVeranstaltungMitTurnierNr($suche->getTurnierNr());
			if ($veranstaltung) {
				return new Suchergebnis($this->toPageArray([$veranstaltung->json()]));
			} else {
				return new Suchergebnis($this->toPageArray([]));
			}
		} else {
			if ($this->clientPersistenceHandler != null) {
				$this->clientPersistenceHandler->store($suche);
			}

			$apiResult = $this->formationswettbewerbResource->findJson(
				$this->convertDate($suche->getVon()), $this->convertDate($suche->getBis()),
				$suche->getPlzVon(), $suche->getPlzBis(),
				$suche->getLandesverbaende(),
				$suche->getLiga(),
				$suche->getTurnierart(),
				intval($suche->getSeite()),
				intval($suche->getAnzahl())
			);

			return new Suchergebnis($apiResult);
		}
	}

	/**
	 * @param FormationswettbewerbSuche $suche
	 * @return BinaryResponse
	 */
	public function exportieren(FormationswettbewerbSuche $suche)
	{
		$this->validateAndFail($suche);

		if ($suche->getTurnierNr() > 0) {
			$response = $this->formationswettbewerbResource->findVeranstaltungMitTurnierNrPdf($suche->getTurnierNr());
			if ($response) {
				return new BinaryResponse($response);
			} else {
				return null;
			}
		} else {
			$response = $this->formationswettbewerbResource->findPdf(
				$this->convertDate($suche->getVon()), $this->convertDate($suche->getBis()),
				$suche->getPlzVon(), $suche->getPlzBis(),
				$suche->getLandesverbaende(),
				$suche->getLiga(),
				$suche->getTurnierart()
			);

			return new BinaryResponse($response);
		}
	}

	/**
	 * @param FormationswettbewerbSuche $suche
	 * @return \Tanzsport\Validation\BindingResult
	 */
	public function validate(FormationswettbewerbSuche $suche)
	{
		return $this->validator->validate($suche);
	}

	private function getRegeln()
	{
		return $this->getAlleParameter()->regeln;
	}

	private function validateAndFail(FormationswettbewerbSuche $suche)
	{
		$br = $this->validate($suche);
		if ($br->isError()) {
			throw new \InvalidArgumentException("Suchparameter sind ungültig!");
		}
	}
}
